/*
 * parsimony.cpp
 *
 *  Created on: Jan 30, 2014
 *      Author: richer
 */

#include "parsimony.h"
#include <cstdlib>
#include "static_sequences.h"

extern StaticSequences sseq;

int parsimony_reference(ResidueType *x, ResidueType *y, ResidueType *z, int size) {
/*
	cerr << endl;
	cerr << "x[] = ";
	for (int i=0; i<16; ++i) {
		cerr << (int) x[i] << " ";
	}
	cerr << endl;
	cerr << "y[] = ";
	for (int i=0; i<16; ++i) {
		cerr << (int) y[i] << " ";
	}
	cerr << endl;
*/
	int mutations = 0;
	for (int i=0; i<size; ++i) {
		z[i] = x[i] & y[i];
		if (z[i] == 0) {
			z[i] = x[i] | y[i];
			++mutations;
		}
	}
/*
	cerr << "z[] = ";
	for (int i=0; i<16; ++i) {
		cerr << (int) z[i] << " ";
	}
	cerr << endl;
	cerr << "mutations=" << mutations << endl;
*/
	return mutations;
}

/**
 * version optimized by compiler
 */
#ifdef __INTEL_COMPILER
__declspec(vector)
#endif
int parsimony_reference_compiler(
		ResidueType * __restrict__ x,
		ResidueType * __restrict__ y,
		ResidueType * __restrict__ z,
		int size) {
#ifdef __INTEL_COMPILER
	int i, mutations = 0;
	__assume_aligned(x, CPU_MEMORY_ALIGNMENT);
	__assume_aligned(y, CPU_MEMORY_ALIGNMENT);
	__assume_aligned(z, CPU_MEMORY_ALIGNMENT);
	__assume(i%CPU_MEMORY_ALIGNMENT==0);
	#pragma simd vectorlength(32)
	for (i=0; i<size; ++i) {
		z[i] = x[i] & y[i];
		if (z[i] == 0) {
			z[i] = x[i] | y[i];
			++mutations;
		}
	}
	return mutations;
#else
	#if __GNU__ > 3 && (__GNUC_MINOR__ > 6)
	x = (ResidueType *) __builtin_assume_aligned(x, CPU_MEMORY_ALIGNMENT);
	y = (ResidueType *) __builtin_assume_aligned(y, CPU_MEMORY_ALIGNMENT);
	z = (ResidueType *) __builtin_assume_aligned(z, CPU_MEMORY_ALIGNMENT);
	#endif

	int i, mutations = 0;
	for (i=0; i<size; ++i) {
		z[i] = x[i] & y[i];
		if (z[i] == 0) {
			z[i] = x[i] | y[i];
			++mutations;
		}
	}
	return mutations;
#endif
}


#ifdef CPU_SSE2_COMPLIANT
#ifdef CPU_ARCHITECTURE_32_BITS
#ifdef CPU_DATA_SIZE_8_BITS
#include "intrinsics/pars_sse2_data_8bits.cpp"
extern "C" { int pars_sse2_32bits_data_8bits(ResidueType *x, ResidueType *y, ResidueType *z, int size); }
#elif CPU_DATA_SIZE_32_BITS
#include "intrinsics/pars_sse2_data_32bits.cpp"
extern "C" { int pars_sse2_32bits_data_32bits(ResidueType *x, ResidueType *y, ResidueType *z, int size); }
#endif
#endif
#ifdef CPU_ARCHITECTURE_64_BITS
#ifdef CPU_DATA_SIZE_8_BITS
#include "intrinsics/pars_sse2_data_8bits.cpp"
extern "C" { int pars_sse2_64bits_data_8bits(ResidueType *x, ResidueType *y, ResidueType *z, int size); }
#elif CPU_DATA_SIZE_32_BITS
#include "intrinsics/pars_sse2_data_32bits.cpp"
extern "C" { int pars_sse2_64bits_data_32bits(ResidueType *x, ResidueType *y, ResidueType *z, int size); }
#endif
#endif
#endif

// names
const char *parsimony_methods_names[] = {
		"",
		"reference",
		"compiler",
#ifdef CPU_SSE2_COMPLIANT
		"sse2",
		"sse_i",
#endif
#ifdef CPU_AVX2_COMPLIANT
		"avx2",
		"avx2_i",
#endif
		NULL
};

// methods
ParsimonyMethodType methods[] = {
		NULL,
		parsimony_reference,
		parsimony_reference_compiler,
#ifdef CPU_SSE2_COMPLIANT
#ifdef CPU_ARCHITECTURE_32_BITS
#ifdef CPU_DATA_SIZE_8_BITS
		pars_sse2_32bits_data_8bits,
		parsimony_sse2_intrinsics,
#elif CPU_DATA_SIZE_32_BITS
		pars_sse2_32bits_data_32bits,
		parsimony_sse2_intrinsics,
#endif
#endif
#ifdef CPU_ARCHITECTURE_64_BITS
#ifdef CPU_DATA_SIZE_8_BITS
		pars_sse2_64bits_data_8bits,
		parsimony_sse2_intrinsics,
#elif CPU_DATA_SIZE_32_BITS
		pars_sse2_64bits_data_32bits,
		parsimony_sse2_intrinsics,
#endif
#endif
#endif
		NULL,
};

int Parsimony::get_nbr_methods() {
	int i=0;
	while (parsimony_methods_names[i] != 0) ++i;
	return i;
}

const char *Parsimony::get_method_name(int n) {
	if ((n == 0) && (n >= get_nbr_methods())) {
		cerr << "error: method " << n << " is undefined" << endl;
		exit(EXIT_FAILURE);
	}
	return parsimony_methods_names[n];
}

int Parsimony::execute(int n, ResidueType *x, ResidueType *y, ResidueType *z, int size) {
	return (*methods[n])(x,y,z,size);
}

int Parsimony::score(DynamicTree::Node *node) {
	if (node->is_leaf()) {
		return 0;
	} else {
		int ls = score(node->l);
		int rs = score(node->r);
		int ts = execute(Parameters::get_instance().method,
				sseq.get_sequence(node->l->n),
				sseq.get_sequence(node->r->n),
				sseq.get_sequence(node->n),
				Parameters::get_instance().taxa_length);
		return ls + rs + ts;
	}
}

int Parsimony::score(DynamicTree& t) {
	return score(t.root);
}

int Parsimony::score(int k, StaticTree::Node *node) {
	int node_id = k;
	int score = 0;

	while (node[node_id].p != -1) {
		int l = node[node_id].l;
		int r = node[node_id].r;
		score += execute(Parameters::get_instance().method,
				sseq.get_sequence(l),
				sseq.get_sequence(r),
				sseq.get_sequence(node[node_id].n),
				Parameters::get_instance().taxa_length);
		++node_id;
	}
	return score;
}

int Parsimony::score(StaticTree& t) {
	return score(Parameters::get_instance().nbr_taxa, t.nodes);
}
