#!/bin/sh

# will generate configuration files
# - src/asm_config.inc
# - src/cpu_config.h
# - src/gpu_config.h

# from parameters/config.params

input="parameters/config.params"
cfg_os=`cat $input | grep "CFG_OS" | cut -d'=' -f2 | tr -d ' '`
cfg_arch=`cat $input | grep "CFG_ARCHITECTURE" | cut -d'=' -f2 | tr -d ' '`
cfg_popcnt=`cat $input | grep "CFG_CPU_POPCOUNT_COMPLIANT" | cut -d'=' -f2 | tr -d ' '`
cfg_vectorize=`cat $input | grep "CFG_CPU_VECTORIZE" | cut -d'=' -f2 | tr -d ' '`
cfg_cmemal=`cat $input | grep "CFG_CPU_MEMORY_ALIGNMENT" | cut -d'=' -f2 | tr -d ' '`
cfg_gmemal=`cat $input | grep "CFG_GPU_MEMORY_ALIGNMENT" | cut -d'=' -f2 | tr -d ' '`
cfg_data_size=`cat $input | grep "CFG_DATA_SIZE" | cut -d'=' -f2 | tr -d ' '`
cfg_gpu_cc=`cat $input | grep "CFG_GPU_COMPUTE_CAPABILITY" | cut -d'=' -f2 | tr -d ' '`

# check parameters
if test "$cfg_os" = "" ; then
	echo "error: variable CFG_OS in $input is not defined"
	exit 1;
fi

if test "$cfg_arch" = "" ; then
	echo "error: variable CFG_ARCHITECTURE in $input is not defined"
	exit 1;
fi

if test "$cfg_cmemal" = "" ; then
	echo "error: variable CFG_CPU_MEMORY_ALIGNMENT in $input is not defined"
	exit 1;
fi

if test "$cfg_gmemal" = "" ; then
	echo "error: variable CFG_GPU_MEMORY_ALIGNMENT in $input is not defined"
	exit 1;
fi

if test "$cfg_popcnt" = "" ; then
	echo "error: variable CFG_CPU_POPCOUNT_COMPLIANT in $input is not defined"
	exit 1;
fi

# check values
if test "$cfg_os" != "linux" -a "$cfg_os" != "windows" ; then
	echo "error: variable CFG_CONFIGURATION in $input "
	echo "should be set to linux or windows"
	echo "current value is: $cfg_os"
	exit 2;
fi

if test "$cfg_arch" != "32" -a "$cfg_arch" != "64" ; then
	echo "error: variable CFG_ARCHITECTURE in $input "
	echo "should be set to 32 or 64"
	echo "current value is: $cfg_arch"
	exit 2;
fi

if test "$cfg_cmemal" != "1" -a "$cfg_cmemal" != "16" -a "$cfg_cmemal" != "32" ; then
	echo "error: variable CFG_CPU_MEMORY_ALIGNMENT in $input "
	echo "should be set to 1 or 16 or 32"
	echo "current value is: $cfg_cmemal"
	exit 2;
fi


if test "$cfg_gmemal" != "1" -a "$cfg_gmemal" != "16" -a "$cfg_gmemal" != "128" ; then
	echo "error: variable CFG_GPU_MEMORY_ALIGNMENT in $input "
	echo "should be set to 1 or 16 or 128"
	echo "current value is: $cfg_gmemal"
	exit 2;
fi

if test "$cfg_popcnt" != "yes" -a "$cfg_popcnt" != "no" ; then
	echo "error: variable CFG_CPU_POPCOUNT_COMPLIANT in $input "
	echo "should be set to yes or no"
	echo "current value is: $cfg_popcnt"
	exit 2;
fi

if test "$cfg_vectorize" != "no" -a "$cfg_vectorize" != "sse2" -a "$cfg_vectorize" != "sse42" -a "$cfg_vectorize" != "avx2" ; then
	echo "error: variable CFG_CPU_VECTORIZE in $input "
	echo "should be set to no, sse2, sse42 or avx2"
	echo "current value is: $cfg_vectorize"
	exit 2;
fi

if test "$cfg_data_size" != "8" -a "$cfg_data_size" != "32"  ; then
	echo "error: variable CFG_DATA_SIZE in $input "
	echo "should be set to 8 or 32"
	echo "current value is: $cfg_data_size"
	exit 2;
fi

if test "$cfg_gpu_cc" != "20" -a "$cfg_gpu_cc" != "21" -a "$cfg_gpu_cc" != "30" -a "$cfg_gpu_cc" != "35" ; then
	echo "error: variable CFG_GPU_COMPUTE_CAPABILITY in $input "
	echo "should be set to 20, 21, 30 or 35"
	echo "current value is: $cfg_gpu_cc"
	exit 2;
fi

# ===============================
# generate files
# ===============================
output_asm="src/asm_config.inc"
output_cpu="src/cpu_config.h"
output_gpu="src/gpu_config.h"
output_mgp="gpu.make"

rm -rf $output_asm $output_cpu $output_gpu $output_mgp

echo "generate configuration files"
echo "- $output_asm"
echo "- $output_cpu"
echo "- $output_gpu"
echo "- $output_mgp"

mkdir -p src/

asm_os=`echo $cfg_os | tr 'a-z' 'A-Z'`
asm_vectorize=`echo $cfg_vectorize | tr 'a-z' 'A-Z'`

echo "%define ASM_OS_$asm_os" >> $output_asm
echo "%define ASM_ARCHITECTURE $cfg_arch" >> $output_asm
echo "%define ASM_CPU_MEMORY_ALIGNMENT $cfg_cmemal" >>$output_asm
if test "$cfg_popcnt" = "yes" ; then
echo "%define ASM_POPCNT_COMPLIANT 1" >>$output_asm
fi
echo "%define ASM_${asm_vectorize}_COMPLIANT 1" >>$output_asm
echo "%define ASM_DATA_SIZE ${cfg_data_size}" >>$output_asm

cpp_os=`echo $cfg_os | tr 'a-z' 'A-Z'`
cpp_arch=`echo $cfg_arch | tr 'a-z' 'A-Z'`
cpp_vectorize=`echo $cfg_vectorize | tr 'a-z' 'A-Z'`

echo "#ifndef CPU_CONFIG_H" >>$output_cpu
echo "#define CPU_CONFIG_H" >>$output_cpu
echo "#define CPU_OS_$cpp_os" >>$output_cpu
echo "#define CPU_ARCHITECTURE_"$cpp_arch"_BITS" >>$output_cpu
echo "#define CPU_MEMORY_ALIGNMENT $cfg_cmemal" >>$output_cpu
echo "#define CPU_${cpp_vectorize}_COMPLIANT" >>$output_cpu
echo "#define CPU_DATA_SIZE_${cfg_data_size}_BITS ${cfg_data_size}" >>$output_cpu
echo "#endif " >>$output_cpu

echo "#ifndef GPU_CONFIG_H" >>$output_gpu
echo "#define GPU_CONFIG_H" >>$output_gpu
echo "#define GPU_MEMORY_ALIGNMENT $cfg_gmemal" >>$output_gpu
echo "#endif " >>$output_gpu

echo "CUDA_SM=-arch sm_${cfg_gpu_cc}" >> $output_mgp

