#!/usr/bin/env python3

# ###################################################################
#
#       Program: tsp_visualizer.py
#        Author: Jean-Michel Richer
#  Organisation: Computer Science Department 
#                Faculty of Science
#                University of Angers
#                2 Boulevard Lavoisier
#                49045 Angers Cedex 01
#                France
#         Email: jean-michel.richer@univ-angers.fr
# Creation date: April, 2021
#  Modification: May, 2021
#
# ###################################################################
# 
# Aim:
#
#    This program is part of a project called TSP Visualizer
#    for Travelling Salesman Problem Visualizer that enables
#    to graphically display a solution of the TSP problem.
#    It is necessary for this to provide a list of cities
#    as well as the distances between these cities and the path
#    (which is a Hamiltonian circuit) to be displayed. One can also
#    provide the positions of the cities on a fictitious map in order 
#    to be able to position these cities on the screen which
#    gives more meaning to the solution.
#
# Objectif :
#
#    Ce programme fait partie d'un projet nommé TSP Visualizer
#    pour Visualiseur du problème du Voyageur de Commerce appelé
#    Travelling Salesman Problem en anglais. Ce programme permet
#    de représenter graphiquement une solution d'une instance
#    de ce problème.
#    Il est nécessaire pour celà de fournir une liste de villes
#    ainsi que les distances qui séparent ces villes et le chemin
#    (qui est un circuit hamiltonien) à afficher. On peut également
#    fournir les positions des villes sur une carte fictive de 
#    manière à pouvoir positionner ces villes sur l'écran ce qui
#    donne plus de sens à la solution.
#
# ###################################################################
#
# License
#
#    This program is a free software you can use, modifiy and 
#    redistribute it for non profitable use. If you use this
#    code please cite the program as TSP Visualizer and the 
#    author's name.
#    Please inform the author of possible bugs or evolution 
#    requests.
#
# Licence
#
#    Ce programme est un logiciel libre que vous pouvez utiliser, 
#    modifier et redistribuer pour un usage non lucratif. Si vous
#    utilisez ce code, merci de bien vouloir citer le nom du projet
#    TSP Visualizer et le nom de l'auteur. 
#    Merci d'informer l'auteur de bogues éventuels ou de demandes 
#    d'évolution.
#
#
# ###################################################################


import io 
import os                       
from os import path
import sys
import math
import numpy as np
import pygame
from pygame.locals import *
import ez_pygame
import re
import color_model as cm
import ez_pygame_fonts_manager as fm
import eztsp_loader

# for clipboard 
import pyperclip

import threading
import solveur
import time


# ===================================================================
# Valeurs par défaut de la fenêtre
# ===================================================================
WINDOW_HEIGHT =  640
WINDOW_WIDTH  = 1280
WINDOW_TITLE_MARGIN = 30
WINDOW_BOTTOM_MARGIN = 40
X_SCALE_DISPLACEMENT = 150
Y_SCALE_DISPLACEMENT = 100

# ===================================================================
# Contenus à afficher
# ===================================================================
WINDOW_TEXT_MENU = "F1 Help - F2 View - F3 List of Cities - F4 About - F10 Quit"

WINDOW_TEXT_HELP = """
Use the following keys :

- F1 for the help menu
- F2 to view path as a graph (graph mode)
- F3 to view path as a list of cities (cities mode)
- F4 for the about window 
- F5 to change color model
- F6 to show / hide cities' names (in view mode F2)
- F7 to show / hide distances (in view mode F2)
- F8 to move the "Distance=" message
- F9 to print path on console and copy to clipboard (in view mode F2)
- F10, Ctrl-C to quit
- F11 to change the size of the window
- F12 to save the path view as an image (in graph mode F2)
- 'L' display list of cities
- 'I' Resolution of the problem based on an iterated local seach algorithm
- 'S' Resolution of the problem based on a simulated annealing algorithm 
- 'K' Resolution of the problem based on call to LKH program (Best Solver)
- 'M' Resolution of the problem based on Minizinc
- 'B' Show or hide background (if a background image is used)

To modify the path in graph mode (F2) :
- Click on a first city and another one to link them
- Right click in graph mode to go back to the previous path
"""

WINDOW_TEXT_ABOUT = """
##################################################################

          Copyright (C) 2021 by Jean-Michel RICHER
          email: jean-michel.richer@univ-angers.fr

##################################################################

This program helps visualize a path for the Traveling Salesman 
problem. The distance is computed and displayed. The path can
be modified by clicking on two cities to link them.

This program can be modified and redistributed for non commercial
purpose.

##################################################################

Please inform the author of possible bugs or evolution requests.
"""

# ===================================================================
#
#   Classe utilisée pour gérer l'affichage du chemin, des villes, des
#   distances entre villes dans la fenêtre pygame mais également le
#   menu d'aide, l'affichage de la liste des villes
#   L'implantation est basée sur la librairie ez_pygame.
#
# ===================================================================

class TSPVisualizer(object):

    """
    CLASS
        Class used to handle all data of the TSP problem and 
        draw a path and its cities in the pygame window as well 
        as the help menu, the list of cities and the about menu.
        The implementation is based on the ez_pygame library.
    """

    # =====================================================
    # Définition et initialisation des données
    # ===================================================== 
    
    def __init__( self, lkh, minizinc ):
    
        """
        WHAT
            Constructor of the application

        HOW
            We initialize all the parameters of the application
                    
        PARAMETERS
            - lkh (string): LKH binary to solve the TSP
            - minizinc (string) : Minizinc parameters
            
        """
        
        # number of cities
        self.count = 0
        self.list_of_cities = []
        self.matrix_of_distances = []
        self.list_of_positions = []
        self.list_cities_text = []
        self.path = []
        self.background = ""
        self.background_image = None
        self.show_background = True
                
        self.list_of_paths = []
        self.show_cities = True
        self.show_distances = False
        self.distance_mode = 0
        
        # define possible windows sizes
        self.window_sizes = [ [WINDOW_WIDTH, WINDOW_HEIGHT],
            (640, 480), (800,600), (1024,768), (768,1024), (1960,1080) ]
        self.window_sizes_index = 0 
        
        self.lkh = lkh
        self.minizinc = minizinc
        
        self.solveur = None
        self.message_solveur = ""
        self.compteur_minizinc = 0

        # message to display temporarily 
        self.message = ""
        self.message_ticks = 0
                            
    # =====================================================
    # Calcul des coordonnées des noms des villes de manière
    # à ce que le nom de la ville apparaissent décalé par 
    # rapport à sa position centrale
    # ===================================================== 
    
    def generate_cities_names_positions( self ):
    
        """
        WHAT
            Define the positions of the names of the cities
            based on the position of each city.
            
        HOW
            Each position of the city name is offset from 
            the position of the city
        """
        
        self.list_cities_text = []
        
        # compute coordinates of barycenter
        X_center, Y_center = 0, 0
        for c in self.list_of_positions:
            X_center += c[0]
            Y_center += c[1]
        
        X_center = int( X_center / self.count )
        Y_center = int( Y_center / self.count )
        
        self.X_center = X_center
        self.Y_center = Y_center
        
        # compute angle of city coordinates to barycenter
        # then move coordinates up, down, left or right
        i = 0
        for c in self.list_of_positions:
            x, y = c[0], c[1]
            d_x = x - X_center
            d_y = y - Y_center
            distance = math.sqrt( d_x * d_x + d_y * d_y )
            angle = math.atan2( d_y, d_x )
            angle_in_degrees = math.degrees( angle ) 
        
            d2 = distance * 1.05
            n_x = X_center + int( d2 * math.cos( angle ) )
            n_y = Y_center + int( d2 * math.sin( angle ) )
            n_x, n_y = c[0], c[1]
        
            if 90.0 < angle_in_degrees < 180.0:
                n_x -= 10
            elif 0.0 < angle_in_degrees < 90.0:
                n_x += 10
        
            if 0.0 < angle_in_degrees < 180.0:
                n_y += 20
            else:
                n_y -= 20
                 
            self.list_cities_text.append( (n_x, n_y) )
            i += 1
            
    # =====================================================
    # Vérifie que les villes sont toutes présentes dans le
    # chemin
    # ===================================================== 
    
    def path_alldifferent( self, a_path ):
    
        """
        WHAT
            Check if all cities are present in the path
            
        HOW
            We check if each city represented by an integer
            is present on the path. This corresponds to an
            AllDifferent constraint. Note that the first
            city has index 1.
            
        PARAMETERS
            - a_path (list of int):     
         
         RETURN
            - True if all city are present
            - False otherwise
            
        """
        
        for i in range(1, len(a_path)+1):
            if not i in a_path:
                raise RuntimeError( "city " + str(i) + " is missing in path" )
                return False
                
        return True

    # =====================================================
    # Changement de repère pour les position des villes
    # de manière à rester dans la fenêtre
    # =====================================================
    
    def scale_positions( self ):
    
        """
        WHAT
            Change positions of cities depending on the size
            of the window. This is done if we don't use a 
            background image.
            
        HOW
            We use a change of scale based on the minimum and
            maximum x and y coordinates of the cities
        """
                
        if len( self.background ) == 0: 
            # modification des positions des villes afin de
            # s'adapter à la taille de la fenêtre d'affichage
            # (changement de repère)
            vx = [] 
            vy = []
            for x,y in self.list_of_original_positions:
                vx.append(x)
                vy.append(y)
                
            x_min = min(vx) 
            x_max = max(vx)
            y_min = min(vy) 
            y_max = max(vy)
                
            XW = WINDOW_WIDTH - X_SCALE_DISPLACEMENT
            YH = WINDOW_HEIGHT - Y_SCALE_DISPLACEMENT

            for i in range( len( self.list_of_positions ) ):
                x = self.list_of_original_positions[i][0]
                y = self.list_of_original_positions[i][1]
            
                x = int( 75 + ( (x - x_min) / (x_max - x_min) * XW) )
                y = int( 50 + ( (y - y_min) / (y_max - y_min) * YH ) )
                self.list_of_positions[i][0] = x
                self.list_of_positions[i][1] = y
        
        else:
            # si on utilise une image de fond (background)
            # on ne modifie pas les positions initiales des
            # villes
            self.list_of_positions= []
            for i in range( len( self.list_of_original_positions ) ):
                x = int( self.list_of_original_positions[i][0] )
                y = int( self.list_of_original_positions[i][1] )
                self.list_of_positions.append( [x,y] )
                
            
        # en fonction des positions des villes on adapte le
        # positionnement des noms des villes qui seront
        # décalés
        self.generate_cities_names_positions()  
    
    
    # =====================================================
    # Lecture et modification des données pour qu'elles
    # puissent être affichées correctement dans la fenêtre
    # =====================================================
    
    def setup_data( self, tsp_loader ):
    
        """
        WHAT 
            setup number of cities, names of cities, matrix of 
            distances between cities, positions of cities 
            from loader 

        PARAMETERS
            - tsp_loader (EZTSPLoader) : class in charge of 
            reading and storing information about the problem 
            to display
        """ 
        
        global WINDOW_WIDTH, WINDOW_HEIGHT
        
        self.tsp_loader = tsp_loader
        
        self.count = tsp_loader.count
                
        self.matrix_of_distances = np.array( tsp_loader.matrix_of_distances )
        
        self.list_of_cities = tsp_loader.list_of_cities
        if len( self.list_of_cities ) == 0:
            for x in range( len( self.count ) ):
                self.list_of_cities.append( "city " + str(x+1) )
        
        self.list_of_original_positions = tsp_loader.list_of_positions 
        
        if len( self.list_of_original_positions ) == 0:
            self.generate_positions()
        else:
            self.list_of_positions = self.list_of_original_positions.copy()
            
            
        # récupération du chemin, si celui-ci est vide on
        # génère un chemin par défaut dans l'ordre d'apparition
        # des villes dans la description du problème
        self.path = tsp_loader.path
        if len( self.path ) == 0:
            self.path = [ x+1 for x in range( self.count ) ]
            
        # modification du chemin, on enlève 1 à chaque indice
        # de ville afin de s'adapter au stockage en C des tableaux
        # qui commence à l'indice 0
        for i in range(self.count):
            self.path[i] -= 1
            self.list_of_cities[ i ] = str(i+1) + ". " + self.list_of_cities[ i ]
        
        # mise à jour de la taille de la fenêtre si cellle-ci
        # est définie
        if len( tsp_loader.window_dimensions ) == 2:
            WINDOW_WIDTH = tsp_loader.window_dimensions[ 0 ]
            WINDOW_HEIGHT = tsp_loader.window_dimensions[ 1 ]
            self.window_sizes[0][0] = WINDOW_WIDTH
            self.window_sizes[0][1] = WINDOW_HEIGHT
        
        # mise à jour de l'image de fond
        self.background = tsp_loader.background
        
        # calcul de la position des villes en fonction de la
        # taille de la fenêtre et de l'utilisation d'un fichier
        # image en fond de fenêtre
        self.scale_positions()
        
        
    # =====================================================
    # Définition du chemin à afficher
    # =====================================================
    
    def set_path( self, path ):
    
        """
        WHAT
            Set path to display
            
        PARAMETERS
            - path (string) : path to display as a comma
            or semicolon separated values   
        """
        
        x = re.split( '[,|; ]+', path ) 
        if len(x) != self.count:
            s = "bad number of cities in path: only " + str( len( x ) ) + " found but "
            s += str( self.count ) + " expected"
            raise ValueError( s )
        self.path = []
        for v in x:
            self.path.append( int(v) )
        
        if not self.path_alldifferent( self.path ):
            raise ValueError( "City is missing or appears twice" )
            
        for i in range(self.count):
            self.path[i] -= 1   
        
        
    # =====================================================
    # Génère les position des villes si elles ne sont pas
    # fournies de manière à ce qu'elles soient sur un 
    # cercle
    # ===================================================== 
        
    def generate_positions( self ):
    
        """
        WHAT
            Generate positions of cities if they were 
            not provided in the .eztsp file
            that describes the problem
            
        HOW
            The positions are put on a circle   
        """
        
        global WINDOW_WIDTH, WINDOW_HEIGHT
        
        angle_delta = 360 / self.count
        angle = 0
        for i in range( self.count ):
            x = WINDOW_TITLE_MARGIN +WINDOW_WIDTH // 2 + int( 200 * math.cos( math.radians( angle ) ) ) 
            y = WINDOW_HEIGHT // 2 + int( 200 * math.sin( math.radians( angle ) ) )
            angle += angle_delta
            self.list_of_original_positions.append( [x,y] )
        
    # =====================================================
    # Affiche les données sous forme de chaine
    # =====================================================
            
    def __str__( self ):
        s = str( self.count ) + "\n"
        s += "==== names ====\n"
        i = 1
        for city_name in self.list_of_cities:
            s += str(i) + "." + city_name + "\n"
            i += 1
        s += "==== distances ====\n"
        s += np.array2string( self.matrix_of_distances ) + "\n"
        s += "==== positions ====\n"
        i = 1
        for x,y in self.list_of_positions:
            s += str( i ) + ". (" + str( x ) +","+str( y ) + ")\n"
            i += 1
                
        return s    
        
    # =====================================================
    # Calculer la distance à partir des données du
    # chemin par défaut stocké dans self.path
    # ===================================================== 
    def compute_distance( self ):
    
        """
        WHAT
            Objective (or score) function that computes
            the distance of the path
            
        HOW
            We add the distances from one city to the next city
            using the self.path variable which is a list of 
            integers    
        
        RETURN
            a float value
        """
        
        distance = 0
        for i in range(1, self.count):
            src = self.list_of_positions[ self.path[i-1] ]
            dst = self.list_of_positions[ self.path[i] ]
            distance += self.matrix_of_distances[ self.path[i-1] ][ self.path[i] ]
        distance += self.matrix_of_distances[ self.path[i] ][ self.path[0] ]
        return distance
    
    
    # =====================================================
    # Dessine le chemin, c'est à dire les villes et les
    # segments de droite qui relient deux villes
    # =====================================================     
    
    def draw_path(self, window):
    
        """
        WHAT
            Draw path using cities positions and names
            
        PARAMETERS
            - window (pygame.window)        
        
        """ 
        
        distance = 0
        
        LINE_COLORS = colors.get( "LINES" )
        
        if self.background_image != None:
            if self.show_background == True:
                window.blit( self.background_image, (0,WINDOW_TITLE_MARGIN) )
        
        # draw lines between cities and compute distance
        for i in range(1, self.count):
            src = self.list_of_positions[ self.path[i-1] ]
            dst = self.list_of_positions[ self.path[i] ]
            color = LINE_COLORS[ i % len(LINE_COLORS) ]
            pygame.draw.line( window, color, (src[0], src[1] + WINDOW_TITLE_MARGIN),
                (dst[0], dst[1] + WINDOW_TITLE_MARGIN), 3 )
            d = self.matrix_of_distances[ self.path[i-1] ][ self.path[i] ]  
            distance += d
            # show distance between cities
            if self.show_distances:
                str_d = str( d )
                x = (src[0] + dst[0]) // 2
                y = (src[1] + dst[1]) // 2 + WINDOW_TITLE_MARGIN
                gui.draw_text( str_d, x, y, colors.get( "CITY_POS" ), None, font_name="pms", center = True )
        
        # trace une ligne entre la dernière ville du chemin et
        # la première
        src = dst
        dst = self.list_of_positions[ self.path[0] ]
        color = LINE_COLORS[ (i+1) % len( LINE_COLORS ) ]
        pygame.draw.line( window, color, (src[0], src[1] + WINDOW_TITLE_MARGIN),
            (dst[0], dst[1] + WINDOW_TITLE_MARGIN), 3 ) 
        distance += self.matrix_of_distances[ self.path[i] ][ self.path[0] ]
        
        # dessine la position de chaque ville sous forme d'un cercle
        # de 5 pixels de rayon
        for x,y in self.list_of_positions:
            y += WINDOW_TITLE_MARGIN
            pygame.draw.circle( window, colors.get( "CITY_POS" ), (x, y), 5 )

        fonts = self.gui.get_font( "pm14" )
        fontl = self.gui.get_font( "pm24" )
        
        # affiche les noms des villes
        if self.show_cities:
            for i in range( 0, self.count ):
                x = self.list_cities_text[ i ][ 0 ]
                y = self.list_cities_text[ i ][ 1 ]
                rendu = fonts.render(self.list_of_cities[i], True, colors.get( "CITY_NAME" ) )
                rect = rendu.get_rect() 
                rect.center = ( x, y + WINDOW_TITLE_MARGIN )
                pygame.draw.rect( window, colors.get( "BG" ), rect )
                window.blit( rendu, rect )
        
        # affiche la distance du chemin en fonction de la position
        # choisie par l'utilisateur
        distance_str = "Distance " + str( int(distance) )
        rendu = fontl.render( distance_str, True, colors.get( "HIGHLIGHT" ) )
        rect = rendu.get_rect() 

        if self.distance_mode == 0: # top left
            rect.move_ip(10, WINDOW_TITLE_MARGIN + 4)
        elif self.distance_mode == 1: # top center
            rect.center = (WINDOW_WIDTH // 2 - 10, WINDOW_TITLE_MARGIN + 18)    
        elif self.distance_mode == 2: # top right
            rect.move_ip(WINDOW_WIDTH - rect[2], WINDOW_TITLE_MARGIN + 4)
        elif self.distance_mode == 3: # center
            rect.center = (WINDOW_WIDTH // 2, (WINDOW_HEIGHT + WINDOW_TITLE_MARGIN) // 2)
        elif self.distance_mode == 4: # bottom left
            rect.move_ip(10, WINDOW_TITLE_MARGIN + WINDOW_HEIGHT - rect[3])
        elif self.distance_mode == 5: # bottom center
            rect.center = (WINDOW_WIDTH // 2, WINDOW_TITLE_MARGIN + WINDOW_HEIGHT - rect[3] // 2)
        elif self.distance_mode == 6: # bottom right
            rect.move_ip(WINDOW_WIDTH - rect[2], WINDOW_TITLE_MARGIN + WINDOW_HEIGHT - rect[3])
            
        pygame.draw.rect( window, colors.get( "BG" ), rect )
        window.blit( rendu, rect )
        
    

    # =====================================================
    # Modification du chemin
    # =====================================================
    
    def modify_current_path( self ):
        # record copy of current path into list
        self.list_of_paths.append( self.path.copy() )
        
        index_src_city = self.path.index( self.src_city_id )
        index_dst_city = self.path.index( self.dst_city_id )

        mode_verbeux = False
        if mode_verbeux == True:
            print( "============================" )
            print( "MODIFY PATH" )
            print( "============================" )
            print( "Initial path=", self.path ) 
            
            print( "city src=", self.src_city_id, index_src_city )
            print( "city dst=", self.dst_city_id, index_dst_city )
        
        self.path.remove( self.dst_city_id )
        if index_dst_city > index_src_city:
            self.path.insert( index_src_city + 1, self.dst_city_id )    
        else:
            self.path.insert( index_src_city - 1, self.dst_city_id )    
        
        if mode_verbeux == True:
            print( "Final path=", self.path )   

    # =====================================================
    # Trouve la ville si on a cliqué sur un point qui
    # correspond à une position de ville
    # =====================================================
    
    def find_city_from_pos_in_graph( self, xs, ys ): 
        i = 0

        for (x,y) in self.list_of_positions:
            if x-10 < xs < x+10:
                if y-10 < ys < y+10:
                    return i
            i = i + 1
    
        return -1
    
    # =====================================================
    # Modification du chemin en fonction des clics souris
    # sur des villes. On clique sur une ville source puis
    # sur une ville destination, dès lors le chemin sera
    # modifié pour la ville de destination apparaissent
    # après la ville source dans le chemin
    # =====================================================
    
    def modify_path_using_mouse( self, xm, ym, city_id ):

        if city_id == -1:
            return  
        
        # if first city then register it    
        if self.mouse_state == 0:
            self.src_city_id = city_id
            self.mouse_state = 1

        # if second city, perform modification          
        elif self.mouse_state == 1:

            self.dst_city_id = city_id
            self.mouse_state = 2
            self.modify_current_path()
            self.mouse_state = 0


    # =====================================================
    # Trouve la ville si on a cliqué sur une ville dans
    # la fenêtre des villes
    # =====================================================
    
    def find_city_from_pos_in_path(self, xs, ys):
        
        Y0 = 40
        X0 = 30
        
        indice_dans_chemin = -1
        
        for i in range( len( self.max_x_list ) - 1 ):
            if self.max_x_list[i] <= xs and xs <= self.max_x_list[i+1]:
                pass

        return indice_dans_chemin


    
    # =====================================================
    # Affiche le titre de la fenêtre en haut ainsi que 
    # le menu en bas de la fenêtre
    # =====================================================
    
    def render_title_and_menu( self, title ):
    
        """
        WHAT
            Window title and menu rendering
            
        HOW
            The window title is positionned on the top of the
            window while the menu is placed at the bottom. The
            title can change but the menu is always the same
            
        PARAMETERS
            - title (string) : title of the window
        """
        
        # titre de la fenêtre
        pygame.draw.rect( gui.window, colors.get( "TITLE_BG" ), [0,0, gui.window_width, WINDOW_TITLE_MARGIN] ) 
        gui.draw_text_center( title, 10, colors.get( "TITLE_TEXT" ), colors.get( "TITLE_BG" ), font_name = "pm24" )
        
        # menu
        gui.draw_text( WINDOW_TEXT_MENU, 30, WINDOW_TITLE_MARGIN + WINDOW_HEIGHT + 4, colors.get( "MENU" ), font_name = "pm22" )
        
    # =====================================================
    # Fixe le message à afficher temporairement ainsi que
    # la durée d'affichage 
    # =====================================================
        
    def set_message( self, s ):
    
        """
        WHAT
            Define the alert message that is displayed for
            a few seconds in the window
            
        HOW
            We define the message to display as well as 
            the duration which is a constant
            
        PARAMETERS  
            s (string) : message to display
        """
        self.message = "\n " + s + " \n"
        self.message_ticks = 300

    # =====================================================
    # Affiche un message de manière temporaire s'il a été
    # défini
    # =====================================================
            
    def render_message( self ):

        """
        WHAT
            Display message for a few seconds
            
        HOW
            The message is displayed during self.message_ticks
            times
        """
        
        if self.message_ticks <= 0:
            self.message = ""
            self.message_ticks = 0
        else:
            gui.draw_text_center( self.message, WINDOW_HEIGHT // 2, colors.get( "TITLE_TEXT" ), colors.get( "TITLE_BG" ), font_name = "pm24" )
            self.message_ticks -= 1
        
    # =====================================================
    # Rendu de la fenêtre "View Mode" qui affiche le
    # chemin, les villes et la distance totale
    # =====================================================
    
    def render_path_as_graph( self ):
    
        """
        WHAT
            Render path as graph
            
        HOW
            We call the self.draw_path() method which does
            all the work    
        """
        
        # titre de la fenêtre
        self.render_title_and_menu( "Path as Graph" )
        
        self.draw_path( self.gui.window )

    # =====================================================
    # Rendu de la fenêtre qui affiche la liste des villes
    # dans l'ordre du chemin
    # =====================================================
    
    def render_path_as_list( self ):
    
        """
        """
        # titre de la fenêtre
        self.render_title_and_menu( "Path as List of Cities" )

        Y0 = 40
        x = 30
        y = Y0
        self.max_x_list = [ ]
        self.max_x_list.append( x )
        max_x = 0
        for i in range(self.count):
            useless, rect = self.gui.draw_text( self.list_of_cities[ self.path[i] ], x, y, colors.get( "TEXT" ), font_name="pm14" )
            max_x = max( max_x, rect[2] )
            y += 20
            if y > WINDOW_HEIGHT - 20:
                y = Y0
                x += max_x + 20
                self.max_x_list.append( x ) 
                max_x = 0
        
        self.max_x_list.append( x + max_x)
        
    # =====================================================
    # Rendu de la fenêtre qui affiche la liste des villes
    # =====================================================

    def render_list_of_cities( self ):
        # titre de la fenêtre
        self.render_title_and_menu( "List of Cities" )

        Y0 = 40
        X0 = 30
        x = X0
        y = Y0
        max_x = 0
        for i in range( self.count ):
            useless, rect = self.gui.draw_text( self.list_of_cities[i], x, y, colors.get( "TEXT" ), font_name="pm14" )
            max_x = max( max_x, rect[2] )
            y += 20
            if y > WINDOW_HEIGHT - 20:
                y = Y0
                x += max_x + 20
                max_x = 0
        
    # =====================================================
    # Rendu de la fenêtre d'aide
    # =====================================================
    
    def render_help( self ):
    
        # titre de la fenêtre
        self.render_title_and_menu( "Help" )
        # texte de l'aide
        gui.draw_text( WINDOW_TEXT_HELP, 15, 25, colors.get( "TEXT" ), font_name="lb16" )


    # =====================================================
    # Rendu de la fenêtre à propos (About)
    # =====================================================
    
    def render_about( self ):

        # titre de la fenêtre
        self.render_title_and_menu( "About" )

        gui.draw_text_center( WINDOW_TEXT_ABOUT, 40, colors.get( "TEXT" ), None, font_name="lb16" )

    
    # =====================================================
    # Initialisations liées à la fonction de rendu
    # =====================================================
    
    def init_rendering_function( self ):
    
        """
        
        """
        
        self.stop_game = False
        self.window_id = 1
        
        # used to change path
        self.mouse_state = 0
        self.src_city_id = -1
        self.dst_city_id = -1
        
        if len(loader.background) != 0:
            self.background_image = pygame.image.load( loader.background ).convert()
            self.scale_positions()
            
    # =====================================================
    # Rendu pour recuit simulé
    # =====================================================
    
    def render_resolution_simulated_annealing( self ):
        x = (WINDOW_WIDTH - 300) // 2
        s = "Simulated Annealing Algorithm"
        gui.draw_text_center( s, 100,  colors.get( "TITLE_BG" ), None, font_name="pm32" )

        s = "  Temperature : {:9.4f}".format( self.solveur.tsp.T )
        gui.draw_text( s, x, 200,  colors.get( "TEXT" ), None, font_name="um24" )
        s = "    Iteration : {:9d}".format( self.solveur.tsp.iteration)
        gui.draw_text( s, x, 250,  colors.get( "TEXT" ), None, font_name="um24" )
        s = "Best distance : {:9.2f}".format( self.solveur.tsp.meilleure_configuration.distance )
        gui.draw_text( s, x, 300,  colors.get( "TEXT" ), None, font_name="um24" )

    # =====================================================
    # Rendu pour recherche locale itérée
    # =====================================================
    
    def render_resolution_iterated_local_search( self ):
        x = (WINDOW_WIDTH - 300) // 2
        s = "Iterated Local Search"
        gui.draw_text_center( s, 100,  colors.get( "TITLE_BG" ), None, font_name="pm32" )
        s = "        Stage : {:9d}".format( self.solveur.tsp.stage)
        gui.draw_text( s, x, 200,  colors.get( "TEXT" ), None, font_name="um24" )

        s = "    Iteration : {:9d}".format( self.solveur.tsp.iteration)
        gui.draw_text( s, x, 250,  colors.get( "TEXT" ), None, font_name="um24" )
        
        s = "      Current : {:9.2f} {:2d}".format( self.solveur.tsp.configuration_courante.distance, self.solveur.tsp.cassures)
        gui.draw_text( s, x, 300,  colors.get( "TEXT" ), None, font_name="um24" )
        
        s = "Best distance : {:9.2f}".format( self.solveur.tsp.meilleure_configuration.distance )
        gui.draw_text( s, x, 350,  colors.get( "TEXT" ), None, font_name="um24" )

    # =====================================================
    # Rendu pour programme LKH
    # =====================================================
    
    def render_resolution_lkh( self ):
        x = (WINDOW_WIDTH - 300) // 2
        s = "Lin-Kernighan Heuristics"
        gui.draw_text_center( s, 100,  colors.get( "TITLE_BG" ), None, font_name="pm32" )
        
        if len(self.message_solveur) != 0:
            gui.draw_text( self.message_solveur, x, 250,  colors.get( "TEXT" ), None, font_name="um24" )

    # =====================================================
    # Rendu pour programme Minizinc
    # =====================================================
    
    def render_resolution_minizinc( self ):
        
        s = "Minizinc"
        gui.draw_text_center( s, 100,  colors.get( "TITLE_BG" ), None, font_name="pm32" )
    
        s = "Searching " + "." * (1 + self.compteur_minizinc)
        x = (WINDOW_WIDTH - 100) // 2
        gui.draw_text( s, x, 200,  colors.get( "TEXT" ), None, font_name="um24" )
        
        if len(self.message_solveur) != 0:
            gui.draw_text( self.message_solveur, x, 250,  colors.get( "TEXT" ), None, font_name="um24" )
            
        self.compteur_minizinc = (self.compteur_minizinc + 1) % 3


    # =====================================================
    # Rendu de la fenêtre de résolution du problème 
    # qui dépend du type de résolution choisi
    # =====================================================
    
    def render_resolution( self, methode ):
        
        """
        Resolution of the problem
        """
        
        if self.solveur == None:
        
            # création du solveur
            if methode == "LKH":
                if len(self.lkh) == 0:
                    self.set_message( "LKH binary not provided" )
                    self.window_id = 2
                    return 
                self.solveur = solveur.Solveur( methode, self.tsp_loader.file_name, self.lkh)
                    
            elif methode == "Minizinc":
                self.solveur = solveur.Solveur( methode, self.tsp_loader.file_name, self.minizinc)  
            
            elif methode == "Descente" or methode == "Recuit":
                self.solveur = solveur.Solveur( methode, self.tsp_loader.file_name, "") 
                
            # lance un thread pour la résolution en parallèle
            self.thread = threading.Thread(target = self.solveur.resoudre, args = () )
            self.thread.start() 
            
        else:
        
            # si le solveur a été précédemment créé, on vérifie s'il a terminé
            # ou non
            if self.solveur.tsp.termine == True:
                gui.draw_text_center( "Finished", 200,  colors.get( "TEXT" ), None, font_name="lb16" )
                self.list_of_paths.append( self.path.copy() )
                
                self.list_of_paths.append( self.path.copy() )
                
                chemin = self.solveur.tsp.meilleure_configuration.chemin
                for i in range( len( chemin ) ):
                    chemin[i] -=1
                self.path =  chemin
                
                self.window_id = 2
                self.solveur.tsp.graphique()
                self.solveur = None
                
                
            else:
                # sinon, on affiche les informations disponibles
                # pour chaque solveur
                
                if methode == "Recuit":
                    self.render_resolution_simulated_annealing()
                    
                elif methode == "Descente":
                    self.render_resolution_iterated_local_search()
                
                elif methode == "LKH":
                    self.render_resolution_lkh()
                
                elif methode == "Minizinc":
                    self.render_resolution_minizinc()
                                        
                time.sleep( 0.5 )
            
            

    # =====================================================
    # Rendu en fonction du mode de visualisation défini
    # par les touches F1, F2, F3, F4, ...   
    # =====================================================
    
    def rendering_function( self, gui ):

        """
        Main function to render the different windows
        """
        
        self.gui = gui          
            
        # rendu du chemin avec villes, noms des villes et
        # distances 
        if self.window_id == 2:
            self.render_path_as_graph()
                                    
        elif self.window_id == 3:
            self.render_path_as_list()
                
        elif self.window_id == 1:
            self.render_help()              

        elif self.window_id == 4:
            self.render_about()     
            
        elif self.window_id == 5:
            self.render_resolution( "Recuit" )
        
        elif self.window_id == 6:
            self.render_resolution( "Descente" )
        
        elif self.window_id == 8:
            self.render_resolution( "LKH" )
            
        elif self.window_id == 9:
            self.render_resolution( "Minizinc" )
                        
        
        elif self.window_id == 7:
            self.render_list_of_cities()
        
        # rendu du message temporaire s'il a été défini
        self.render_message()   
            


    # =====================================================
    # Réalise une capture d'écran et effectue la
    # sauvegarde
    # =====================================================
    
    def screenshot( self ):
    
        """
        Screenshot of path
        """
        
        scrsht_dir = "screenshots"
        scrsht_file = "screenshots" + os.path.sep + ".counter"
        
        try:
            if not path.isdir( scrsht_dir ):
                os.mkdir( scrsht_dir )

            index = 1
            if not path.exists( scrsht_file ):
                file = open( scrsht_file, "w+" )
                file.write( "1" )
                file.close()
            else:
                file = open( scrsht_file, "r" )
                line = file.read()
                file.close()
                index = len(line) + 1
                file = open( scrsht_file, "a" )
                file.write( "1" )
                file.close()
            
            image_file_name = scrsht_dir + os.path.sep + "image_" + str(index) + ".png"
            pygame.image.save( gui.window, image_file_name )
            self.message = "Image saved in " + image_file_name
            self.message_ticks = 100
                
        except (RuntimeError, IOError)  as err:
            self.message = "Error occurred"
            self.message_ticks = 100
            
    # =====================================================
    # Gère les événements
    # les événements de base gérés par toutes les fenêtres
    # sont les suivants :
    # - Ctrl-C ou F-10 ou clic sur x de la fenêtre : on
    #   quitte le programme
    # - F1 revenir à l'écran principal d'affichage du chemin
    # - F2 fenêtre d'affichage la liste des villes
    # - etc voir la variable WINDOW_TEXT_HELP qui décrit
    # les fonctionnalités
    # =====================================================

    def events_handler( self, gui ):
    
        """
        """
        
        global WINDOW_WIDTH, WINDOW_HEIGHT
        
        for event in pygame.event.get():
        
            # quit
            if event.type == QUIT:
                gui.stop = True
                return 
    
            # quit          
            if event.type == KEYDOWN:
                # Ctrl-C to leave game
                if event.key == pygame.K_c and pygame.key.get_mods() & pygame.KMOD_CTRL:
                    gui.stop = True
                    return 
                
                # resize
                if event.key == pygame.K_F11:
                    self.window_sizes_index = (self.window_sizes_index + 1) % len( self.window_sizes )
                    WINDOW_WIDTH = self.window_sizes[ self.window_sizes_index ][ 0 ]
                    WINDOW_HEIGHT = self.window_sizes[ self.window_sizes_index ][ 1 ]
                    gui.window_width = WINDOW_WIDTH
                    gui.window_height = WINDOW_HEIGHT
                    gui.window = pygame.display.set_mode( (WINDOW_WIDTH, WINDOW_TITLE_MARGIN + WINDOW_HEIGHT + WINDOW_BOTTOM_MARGIN) );
                    self.scale_positions()
                    self.generate_cities_names_positions()
                    s = " size " + str( WINDOW_WIDTH ) + "x" + str( WINDOW_HEIGHT ) + " "
                    self.set_message( s )
                    return
                    
                # quit  
                if event.key == pygame.K_F10:
                    gui.stop = True
                    return
                
                # help window   
                if event.key == K_F1:
                    self.window_id = 1
                    return None
                    
                # path as graph
                if event.key == K_F2:
                    self.window_id = 2
                    return None
                    
                # path as list of cities    
                if event.key == K_F3:
                    self.window_id = 3      
                    return None
                
                # about window  
                if event.key == K_F4:
                    self.window_id = 4      
                    return None

                # recherche avec recuit simulé
                if event.key == K_s:
                    if self.window_id == 5:
                        # stoppe la recherche
                        if self.solveur != None:
                            self.solveur.tsp.termine = True
                            time.sleep(1)
                    else:
                        self.window_id = 5
                        time.sleep(1)
                    return None

                # recherche avec recherche locale itérée
                if event.key == K_i:
                    if self.window_id == 5:
                        # stoppe la recherche
                        if self.solveur != None:
                            self.solveur.tsp.termine = True
                            time.sleep(1)
                    else:
                        time.sleep(1)
                        self.window_id = 6
                    return None
                    
                if event.key == K_l:
                    self.window_id = 7
                    return None
                
                # LKH   
                if event.key == K_k:
                    self.window_id = 8
                    return None 
                
                # Minizinc  
                if event.key == K_m:
                    self.window_id = 9
                    return None 
                    
                if event.key == K_b:
                    self.show_background = not self.show_background
                        
                # change color model
                if event.key == K_F5:
                    colors.next()
                    self.set_message( "Color model: " + colors.model_name )
                    gui.background_color = colors.get( "BG" ) 
                    return None

                # show or hide cities' names
                if event.key == K_F6:
                    self.show_cities = not( self.show_cities )  
                    return None
                
                # show or hide distances between cities
                if event.key == K_F7:
                    self.show_distances = not( self.show_distances )    
                    return None
                    
                # change position of string that reports the distance   
                if event.key == K_F8:                   
                    self.distance_mode = ( self.distance_mode + 1 ) % 7
                    
                # go back to view mode  
                if event.key == K_ESCAPE:
                    self.window_id = 2
                    return None
                    
                # print path on terminal and copy to clipboard
                if event.key == K_F9:
                    s = "="*40
                    s += "\nCurrent path\n"
                    s += "="*40 + "\n"
                    s +=  str(self.path[0]+1)
                    for i in range( 1, len( self.path ) ):
                        s += ", " + str( self.path[i] + 1 )
                    s += "\n"
                    s += "distance=" + str( self.compute_distance() )
                    print(s)
                    pyperclip.copy( s )
                    dummy = pyperclip.paste()
                
                if event.key == K_F12:
                    if self.window_id == 2:
                        self.screenshot()
                        
                    
            # en cas de clic souris on tente de déterminer si on a
            # cliqué sur une ville  
            if event.type == MOUSEBUTTONDOWN:
            
                if self.window_id == 2:
                    # si sur la fenêtre du graphe du chemin
                    
                    (button_1, button_2, button_3) = pygame.mouse.get_pressed()

                    if button_1 == 1:
                        mouse = pygame.mouse.get_pos()  
                        x_mouse, y_mouse = mouse[0], mouse[1] - WINDOW_TITLE_MARGIN
                        city_id = self.find_city_from_pos_in_graph( x_mouse, y_mouse ) 
                        self.modify_path_using_mouse( x_mouse, y_mouse, city_id )
                    
                    # clic droit, on remplace le chemin actuel par le chemin
                    # précédent s'il existe 
                    if button_3 == 1:
                        if len( self.list_of_paths ) != 0:
                            self.path = self.list_of_paths.pop()
                        #print( self.path )
            
                elif self.window_id == 3:
                    # si sur la fenêtre des villes
                    
                    (button_1, button_2, button_3) = pygame.mouse.get_pressed()
                    if button_1 == 1:
                        mouse = pygame.mouse.get_pos()  
                        x_mouse, y_mouse = mouse[0], mouse[1] - WINDOW_TITLE_MARGIN
                        city_id = self.find_city_from_pos_in_list( x_mouse, y_mouse ) 
                    
                else:
                    return None 
                            
        # return (valeur nulle)
        return None     
    
    # =====================================================
    # =====================================================
        
    def quit_function( self ):
    
        """
        WHAT
            Quit interface
            
        HOW
            If a solver is searching for a solution then we must
            first stop it.
        """
        
        if self.solveur != None:
            self.solveur.tsp.termine = True
            time.sleep( 0.5 )
        
# 
# ===================================================================
# Main program
# ===================================================================
#

arg_file_name = "data/usa_13.eztsp"
arg_lkh_binary=""
arg_minizinc_params=""
arg_path = ""
visu = None

HELP_ARGUMENTS_MESSAGE = """
Program arguments are:

-h or --help 
    to get this message

-i file or --input=file
    to specify input file

-k file or --lkh=file
    name of LKH binary to solve the problem using the effective 
    implementation of the Lin-Kernighan heuristic 

-m solver,time or --minizinc=solver,time
    name of the solver that will be used by Minizinc to solve 
    the problem followed by the maximum time allocated to the 
    resolution. For example 'chuffed,60' to use the chuffed
    solver during 60 seconds
    
-p string or --path=string
    to specify path to display that will overide path of the
    input file
    
    
"""

#
# gestion des arguments du programme
#

if len(sys.argv) >= 1:
    try:
        i = 1
        while i < len( sys.argv ):
            arg = sys.argv[i] 
            if arg == "-h" or arg == "--help":
                print( HELP_ARGUMENTS_MESSAGE )

            elif arg == "-i": 
                i += 1
                arg_file_name = sys.argv[i]
            elif arg.startswith( "--input=" ):
                a = sys.argv[i].split( "=" )
                arg_file_name = a[1]
                
            elif arg =="-k":
                i += 1
                arg_lkh_binary = sys.argv[i]
            elif arg.startswith( "--lkh=" ):
                a = sys.argv[i].split( "=" )
                arg_lkh_binary = a[1]   

            elif arg =="-m":
                i += 1
                arg_minizinc_params = sys.argv[i]
            elif arg.startswith( "--minizinc=" ):
                a = sys.argv[i].split( "=" )
                arg_minizinc_params = a[1]  
                
            elif arg == "-p":
                i += 1
                arg_path = sys.argv[i]
            elif arg.startswith( "--path=" ):
                a = sys.argv[i].split( "=" )
                arg_path = a[1] 
                arg_path.replace('"', '')
                
            else:
                raise ValueError( "Unexpected argument : " + arg )
            
            i += 1  
            
    except:
        print( "="*40 )
        print( sys.exc_info()[1] )
        print( "="*40 )
        print( HELP_ARGUMENTS_MESSAGE )
        sys.exit( 1 )
        
try:
    loader = eztsp_loader.EZTSPLoader( arg_file_name )
    error_message = loader.load()
    if len( error_message ) != 0:
        raise RuntimeError( error_message )
        
    if len( arg_lkh_binary ) != 0:  
        executable = os.access( arg_lkh_binary, os.X_OK )
        if not executable:
            raise RuntimeError( "LKH binary '" + arg_lkh_binary + "' is not valid" )
        
        
    visu = TSPVisualizer( arg_lkh_binary, arg_minizinc_params ) 
    visu.setup_data( loader )
    
    if len(arg_path) > 0:
        visu.set_path( arg_path )
    #print(visu)
    
except RuntimeError as e:
    print( "=" * 40 )
    print( "ERROR: {0}".format( e ) )
    print( "=" * 40 )


#
# Define color models
# For this application there are 9 different colors defined as
# - BG or BackGround
# - TITLE_BG: background for a title 
# - TITLE_TEXT: foreground color of text in a title
# - TEXT: normal text
# - MENU: color of the menu items
# - CITY_NAME: color of city when displayed in graph mode
# - CITY_POS: color of circle that represents the position of a city
# - HIGHLIGHT: color of text when we need to highlight some part 
# - LINES: list of colors for lines between cities when in graph mode

colors = cm.ColorModel( "BG|TITLE_BG|TITLE_TEXT|TEXT|MENU|CITY_NAME|CITY_POS|HIGHLIGHT|LINES" )

"""
Light color model is based on a white (255,255,255) background
Dark color model is based on a black (0,0,0) background
Grey color model is based on a light grey (0xD9,0xDD,0xDC) background
"""     
colors.add("light color", [(255,255,255),
    (186, 74, 0),
    (255, 255, 255),
    (96,96,96),
    (31, 97, 141),
    (0,128,0),
    (255, 104, 31),
    (169, 50, 38),
    [(32,32,132),(128,28,128),(164,64,64),(60,160,160)]
])

colors.add("dark color", [(0,0,0), 
    (128,128,128), 
    (255,255,255),
    (164,164,164),
    (31, 97, 141),
    (0,128,0),
    (255, 104, 31),
    (169, 50, 38),
    [(32,32,132),(128,28,128),(164,64,64),(60,160,160)]
])

colors.add("grey grey", [(0xD9,0xDD,0xDC), 
    (0x87, 0x7F, 0x7D), 
    (255,255,255),
    (0x77,0x7D,0x7E),
    (0x22, 0x20, 0x21),
    (0x54,0x4C,0x4A),
    (0x97, 0x97, 0x8F),
    (0x86, 0x8F, 0x87),
    [(32,32,32),(128,128,128),(64,64,64),(160,160,160)]
])

colors.add("grey color", [(0xD9,0xDD,0xDC), 
    (186, 74, 0), 
    (255,255,255),
    (0x77,0x7D,0x7E),
    (0x22, 0x20, 0x21),
    (0x54,0x4C,0x4A),
    (186, 74, 0),
    (0x86, 0x8F, 0x87),
    [(32,32,32),(128,128,128),(64,64,64),(160,160,160)]
])

    
colors.set_model( "light color" )

#
# Create main window
#
gui = ez_pygame.EZPygame( "TSP Visualizer", WINDOW_WIDTH, WINDOW_TITLE_MARGIN + WINDOW_HEIGHT + WINDOW_BOTTOM_MARGIN, colors.get( "BG" ) ) 

fonts_definition = {
    "pm" : { "ttf_file": "fonts/PermanentMarker-Regular.ttf", "sizes": [10,12,14,16,18,20,22,24, 32] },
    "lb" : { "ttf_file": "fonts/LiberationSans-Regular.ttf",  "sizes": [10,12,14,16,18,20,22,24] },
    "um" : { "ttf_file": "fonts/UbuntuMono-R.ttf",  "sizes": [16,18,20,22,24] },
}

fma = fm.FontsManager( fonts_definition )
gui.set_font_manager( fma )



#
# Initialize user defined rendering function
#
visu.init_rendering_function()

#
# Call main function 
#
gui.render( visu.rendering_function, visu.events_handler, visu.quit_function )


