# ###################################################################
#
#       Program: tsp_recuit_simule.py
#        Author: Jean-Michel Richer
#  Organisation: Computer Science Department, 
#                University of Angers,
#	             France
#         Email: jean-michel.richer@univ-angers.fr
# Creation date: April, 2021
#  Modification: May, 2021
#
# ###################################################################
# 
# Aim:
#
#    This program is an implementation of a Simulated Annealing 
#    algorithm used to find a solution to the Travelling Salesman 
#    problem. 
#
# Objectif :
#
#    Ce programme repose sur l'implantation d'un algorithme de
#    Recuit Simulé afin de résoudre le problème du Voyageur 
#    de Commerce.
#
# ###################################################################
#
# License
#
#    This program is a free software you can use, modifiy and 
#    redistribute it for non profitable use. If you use this
#    code please cite the program as TSP Visualizer and the 
#	 author's name.
#    Please inform the author of possible bugs or evolution 
#    requests.
#
# Licence
#
#    Ce programme est un logiciel libre que vous pouvez utiliser, 
#    modifier et redistribuer pour un usage non lucratif. Si vous
#    utilisez ce code, merci de bien vouloir citer le nom du projet
#    TSP Visualizer et le nom de l'auteur. 
#    Merci d'informer l'auteur de bogues éventuels ou de demandes 
#    d'évolution.
#
# ###################################################################

import random
import sys
import math
import numpy as np
import matplotlib.pyplot as plt
import tsp_path as path


# ===================================================================
# Les données en entrée sont les suivantes :
# - une liste des villes sous forme de liste de chaines de caractères
# - les distances entre les villes sous forme d'une matrice d'entiers
#   (en l'occurrence il s'agit d'une liste de liste d'entiers)
# ===================================================================



# -----------------------------------------------------
# Redéfinition de la fonction exponentielle
# -----------------------------------------------------
def exponential( x ):
    if x <= -100.0:
        return 0.0
    if x >= 1.0:
        return 1.0
    else:
        return math.exp( x )



        
# ===================================================================
#	On définit une class TSP pour stocker et manipuler les données
#	et résoudre le problème
#	
# ===================================================================

class TSPRecuitSimule(object):

	"""
	Class used to solve the problem that implements a Simulated
	Annealing algorithm
	"""
	
	# -----------------------------------------------------
	# Le constructeur modifie les données de manière à
	# utiliser des indices qui commencent à 1 et non à 0
	#
	# On modifie la matrice des distances en ajoutant une
	# ligne et une colonne de zéros, puis on convertit tous
	# les 0 en une distance égale à la valeur maximum trouvée
	# dans la matrice + 1, ce qui permettra de rechercher une
	# distance minimum sans obtenir de zéro
	# -----------------------------------------------------
	def __init__( self, villes, distances ):
		self.nbr_villes = len( villes )
		self.villes = villes.copy()
		self.villes.insert( 0, "-------" )

		self.distances = np.array( distances )
		colonne = np.zeros( self.nbr_villes ).reshape( 1, self.nbr_villes )
		self.distances = np.insert( self.distances, 0, colonne, axis = 1 )
		ligne = np.zeros( self.nbr_villes + 1 ).reshape( 1, self.nbr_villes + 1 )
		self.distances = np.insert( self.distances, 0, ligne, axis = 0 )
		maximum = np.max( self.distances )
		self.distances[ self.distances == 0 ] = maximum + 1

		# vmode verbeux par défaut
		self.verbeux = True
		# 
		self.termine = False

	# -----------------------------------------------------
	# Définition de la fonction objectif
	# -----------------------------------------------------
	def f_objectif( self, chemin ):
		distance_parcourue = 0
		for i in range( 1, self.nbr_villes ):
		    distance_parcourue += self.distances[ chemin[i-1] ][ chemin[i] ]
		    
		distance_parcourue += self.distances[ chemin[ self.nbr_villes-1] ][ chemin[0] ]    
		return distance_parcourue  
    
	def perturbe1( self, configuration ):
	
		if self.verbeux:
			print( "\n" + "="*40 )
			print( "Perturbation" )
			print( "="*40 )
		# échanger des villes
		mini = int( self.nbr_villes * 0.1)
		maxi = int( self.nbr_villes * 0.3)
		
		nbr_de_villes_a_echanger = random.randint( mini, maxi )

		if self.verbeux:		
			print("number of cities to modify: ", nbr_de_villes_a_echanger)
		while nbr_de_villes_a_echanger != 0:
			x = random.randint( 0, self.nbr_villes - 1 )
			y = random.randint( 0, self.nbr_villes - 1 )
			while y == x:
				y = random.randint( 0, self.nbr_villes - 1 )
				
			v1 = configuration.chemin[ x ]	
			v2 = configuration.chemin[ y ]	
			configuration.chemin[ x ] = v2
			configuration.chemin[ y ] = v1	
			nbr_de_villes_a_echanger -= 1

		configuration.distance = self.f_objectif( configuration.chemin )
	
		
		
	def ville_la_plus_proche( self, ville ):
		"""
		retourne la ville la plus proche de 'ville' 
		et la distance correspondante
		"""
		distance_ville_lpp = self.distances[ 0 ][ 0 ]
		ville_lpp = -1
		
		for i in range( 1, self.nbr_villes + 1 ):
			if self.distances[ ville ][ i ] < distance_ville_lpp:
				distance_ville_lpp = self.distances[ ville ][ i ]
				ville_lpp = i
			
		return ville_lpp, distance_ville_lpp
		
			
	def perturbe2( self, configuration ):
	
		if self.verbeux:
			print( "\n" + "="*40 )
			print( "Perturbation" )
			print( "="*40 )

		chemin = configuration.chemin
		
		# on recherche l'indice dans le chemin de la distance
		# maximum entre deux villes
		d_max_e2v = 0
		indice_d_max_e2v = -1
		
		for i in range( 0, self.nbr_villes ):
			d = self.distances[ chemin[i-1] ][ chemin[i] ]
			if d > d_max_e2v:
				d_max_e2v = d
				indice_d_max_e2v = i-1
						
		# trouver une ville plus proche
		ville_precedente = chemin[ indice_d_max_e2v ]
		ville_suivante = chemin[ (indice_d_max_e2v + 1) % self.nbr_villes ]
		vlpp, dvlpp = self.ville_la_plus_proche( ville_precedente )
		
		chemin.remove( vlpp )
		chemin.insert( (indice_d_max_e2v + 1) % self.nbr_villes, vlpp)
		chemin.remove( ville_suivante )
		
		vlpp, dvlpp = self.ville_la_plus_proche( ville_suivante )
		i = chemin.index( vlpp )
		chemin.insert( i, ville_suivante )
		
		point_de_coupure = random.randint( 1, len( chemin )-1 )
		
		chemin = chemin[point_de_coupure:] + chemin[:point_de_coupure]
		
		configuration.distance = self.f_objectif( configuration.chemin )
		
		if self.verbeux:
			print( "chemin = ", chemin, configuration.distance )
		
	# -----------------------------------------------------
	# Génération de voisins en échangeant des villes deux
	# à deux. Seules les configurations voisines amélio-
	# rantes sont gardées puis triées en ordre croissant.
	# En fonction du nombre d'itérations on élimine 
	# certaines configurations :
	# - si le nombre d'itérations est inférieu à 1000, on
	#   ne garde que les dix premières configurations 
	#   améliorantes
	# - au dela de 5000 itérations on ne garde que la 
	#   première configuration améliorante
	# -----------------------------------------------------	
	def voisinage( self, configuration, iteration ):

		# liste des voisins améliorants
		l = [ ]
		
		distance_a_ameliorer = configuration.distance 
		
		#mini = distance_a_ameliorer
		
		for i in range( 0, self.nbr_villes-1):
			for k in range( i+1, self.nbr_villes):
				if i > self.nbr_villes or k > self.nbr_villes:
					continue
		        
				nouvelle_configuration = configuration.copy()
				# échange les villes aux indices i et k
				ville_1 = nouvelle_configuration.chemin[ i ]
				ville_2 = nouvelle_configuration.chemin[ k ]
				nouvelle_configuration.chemin[ i ] = ville_2
				nouvelle_configuration.chemin[ k ] = ville_1
				nouvelle_configuration.distance = self.f_objectif( nouvelle_configuration.chemin )
				l.append( nouvelle_configuration )
		
		for i in range(0, self.nbr_villes-1):
			for j in range( i + 2, self.nbr_villes + 1):
				nouvelle_configuration = configuration.copy()
				ville = configuration.chemin[ i ]
				nouvelle_configuration.chemin.remove( ville )
				nouvelle_configuration.chemin.insert( j - 1, ville)
				nouvelle_configuration.distance = self.f_objectif( nouvelle_configuration.chemin )
		        
				if nouvelle_configuration.distance < distance_a_ameliorer:
					l.append( nouvelle_configuration )
					        
		# tri des configuration en ordre croissant de la
		# distance parcourue        
		l.sort( key=lambda cfg : cfg.distance )

		return l
    
	# -----------------------------------------------------
	# Méthode de résolution basée sur le recuit simulé
	# -----------------------------------------------------
	def resolution(self, t_i = 100.0, t_f = 0.001, alpha = 0.99):
	
		self.termine = False
		
		self.liste_evolution_courante = []
		self.liste_evolution_meilleur = []
		
		#
		# Génération de la configuration initiale
		#
		villes_dans_le_desordre = [ x for x in range(1, self.nbr_villes+1) ]
		random.shuffle( villes_dans_le_desordre )
		self.configuration_courante = path.Path( villes_dans_le_desordre, self. f_objectif( villes_dans_le_desordre ) )

		if self.verbeux:
			print( "Configuration initiale=", self.configuration_courante )
		
		self.iteration = 1
		
		#
		# Paramètres de l'algorithme
		#
		
		# température initiale
		T_initiale = t_i
		# température finale
		T_finale   = t_f
		# facteur de refroidissement (alpha)
		facteur_de_refroidissement = alpha
		
		self.T = T_initiale
		self.meilleure_configuration = self.configuration_courante.copy()

		columns = "  temp.  |iter.| best | curr. | neigh. nbr,min,max | sel |delta |  e   |  u   |sel" 
		separator = "-" * 85
	
		if self.verbeux:
			print(separator)
			print(columns)
			print(separator)
		
		nbr_consecutif_meilleur = 0
		
		# Tant qu'on a pas atteint la température finale
		while self.T > T_finale:
			
			if self.termine:
				break
				
			meilleur_avant = self.meilleure_configuration.distance

			if self.verbeux:			
				str_T  = "{:9.4f}".format( self.T )
				str_it = "{:5d}".format( self.iteration )
				str_cc = "{:6d}".format( int( self.configuration_courante.distance ) )
				str_mc = "{:6d}".format( int( self.meilleure_configuration.distance ) )
				print(str_T + "|" + str_it + "|" + str_mc + "|" + str_cc + " | ", end='')
			

			# rechercher des voisins améliorants
			l_voisins_ameliorants = self.voisinage( self.configuration_courante, self.iteration )
			
			amelioration = False
			if l_voisins_ameliorants[0].distance < self.configuration_courante.distance:
				i = 0
				amelioration = True
			else:
				i = random.randint(0, len( l_voisins_ameliorants )-1 )
				
			configuration_voisine = l_voisins_ameliorants[ i ]
			
			if self.verbeux:
				str_cv = "{:5d}".format( int( configuration_voisine.distance ) )
				str_lv = "{:5d}".format( len( l_voisins_ameliorants ) )
				str_lp = "{:6d}".format( int( l_voisins_ameliorants[ 0 ].distance) ) 
				str_ld = "{:6d}".format( int( l_voisins_ameliorants[ len( l_voisins_ameliorants ) - 1 ].distance) ) 
				
				print( str_lv + "," + str_lp + "," + str_ld + "|" + str_cv + "|", end='' )


			delta_f = configuration_voisine.distance - self.configuration_courante.distance

			str_u   = "      "
			str_exp = "      "
			str_sel = " no"
			
			e = exponential( float( -delta_f ) / float( self.T ) )
			str_exp = "{:.4f}".format( e )
			
			# si le voisin est meilleur que la configuration courante
			#    on le garde
			# sinon 
			#    on prend un voisin qui détériore la solution
			#    avec une certaine probabilité qui diminue à
			#    mesure que la température diminue
			if delta_f < 0.0:
				self.configuration_courante = configuration_voisine
			else:
				u = random.random()
				e = exponential( float( -delta_f ) / float( self.T ) )
				    
				str_u = "{:.4f}".format( u )
				str_exp = "{:.4f}".format( e )
				
				if (e > u):
				    self.configuration_courante = configuration_voisine
				    str_sel = "yes"
				    
			str_delta_f = "{:6d}".format( int( delta_f ) )
			if self.verbeux:
				print( str_delta_f+"|"+str_exp+"|"+str_u+"|"+str_sel )

			
			if  self.configuration_courante.distance < self.meilleure_configuration.distance:
				self.meilleure_configuration = self.configuration_courante.copy()
			
			# diminution de la température	
			self.T = facteur_de_refroidissement * self.T    
			self.iteration += 1
			
			if (self.iteration % 30) == 0:
				if self.verbeux:
					print( separator )
					print( columns )
					print( separator )

			self.liste_evolution_courante.append( self.configuration_courante.distance )
			self.liste_evolution_meilleur.append( self.meilleure_configuration.distance )
			
			meilleur_apres = self.meilleure_configuration.distance
			if meilleur_avant == meilleur_apres and not( amelioration ):
				nbr_consecutif_meilleur += 1
				if nbr_consecutif_meilleur == 30:

					self.configuration_courante = self.meilleure_configuration.copy()
					if random.random() > 0.75:
						self.perturbe2( self.configuration_courante )
					else:
						self.perturbe1( self.configuration_courante )
						
					nbr_consecutif_meilleur = 0
			else:
				nbr_consecutif_meilleur = 0
		
		self.termine = True
								
		return self.meilleure_configuration
				
	# -----------------------------------------------------
	# Affichage du chemin ainsi que des distances 
	# parcourues et de la distance totale
	# -----------------------------------------------------
	
	def print(self, chemin):
	
		print()
		print( "------------------------------------------" )
		print( "         ville            | dist. | totale" )
		print( "------------------------------------------" )
		distance_totale = 0
		print( "{0:25s} | {1:5d} | {2:5d}".format( self.villes[ chemin[0] ], 0, 0) )
		for ville in range( 1, len( chemin ) ):	
			distance = self.distances[ chemin[ville-1] ][ chemin[ville] ]
			distance_totale += distance
			print( "{0:25s} | {1:5d} | {2:5d}".format( self.villes[ chemin[ville] ], int( distance ), int( distance_totale) ) )
		
		distance = self.distances[ chemin[ville] ][ chemin[0] ]
		distance_totale += distance
		print( "{0:25s} | {1:5d} | {2:5d}".format( self.villes[ chemin[0] ], int(distance) , int( distance_totale) ) )
		print()
		print( "chemin=",chemin )
		print()
	
	# -----------------------------------------------------
	# Affiche un graphique de l'évolution des scores 
	# (distances) de la configuration courante et de la
	# meilleure configuration
	# -----------------------------------------------------	
	def graphique( self ):
		x = range( len( self.liste_evolution_courante ) )
		plt.grid()
		plt.plot( x, self.liste_evolution_courante, label='current path' )
		plt.plot( x, self.liste_evolution_meilleur, label='best path' )
		plt.xlabel('iterations')
		plt.legend()
		plt.title('Evolution of Simulated Annealing')
		plt.show()
			

