# ###################################################################
#
#       Program: tsp_recherche_locale.py
#        Author: Jean-Michel Richer
#  Organisation: Computer Science Department, 
#                University of Angers,
#	             France
#         Email: jean-michel.richer@univ-angers.fr
# Creation date: April, 2021
#  Modification: May, 2021
#
# ###################################################################
# 
# Aim:
#
#    This program is an implementation of a Local Search algorithm
#    used to find a solution to the Travelling Salesman Problem. 
#    
#
# Objectif :
#
#    Ce programme repose sur l'implantation d'un algorithme de
#    Recherche Locale afin de résoudre le problème du Voyageur 
#    de Commerce.
#
# ###################################################################
#
# License
#
#    This program is a free software you can use, modifiy and 
#    redistribute it for non profitable use. If you use this
#    code please cite the program as TSP Visualizer and the 
#	 author's name.
#    Please inform the author of possible bugs or evolution 
#    requests.
#
# Licence
#
#    Ce programme est un logiciel libre que vous pouvez utiliser, 
#    modifier et redistribuer pour un usage non lucratif. Si vous
#    utilisez ce code, merci de bien vouloir citer le nom du projet
#    TSP Visualizer et le nom de l'auteur. 
#    Merci d'informer l'auteur de bogues éventuels ou de demandes 
#    d'évolution.
#
# ###################################################################


import sys
import random
import numpy as np
import matplotlib.pyplot as plt
import tsp_path as path

# ===================================================================
# Les données en entrée sont les suivantes :
# - une liste des villes sous forme de liste de chaines de caractères
# - les distances entre les villes sous forme d'une matrice d'entiers
#   (en l'occurrence il s'agit d'une liste de liste d'entiers)
# ===================================================================

# ===================================================================
#	On définit une class TSP pour stocker et manipuler les données
#	et résoudre le problème
#	
# ===================================================================
class TSPRechercheLocale( object ):
	"""
	Class used to solve the problem that implements a Local Search
	algorithm
	"""
	
	# -----------------------------------------------------
	# Le constructeur modifie les données de manière à
	# utiliser des indices qui commencent à 1 et non à 0
	#
	# On modifie la matrice des distances en ajoutant une
	# ligne et une colonne de zéros, puis on convertit tous
	# les 0 en une distance égale à la valeur maximum trouvée
	# dans la matrice + 1, ce qui permettra de rechercher une
	# distance minimum sans obtenir de zéro
	# -----------------------------------------------------
	
	def __init__( self, villes, distances ):
		"""
		Constructor
		
		Parameters
		----------
		villes : list of string
		  list of cities names
		distances : list of list of int
		  matrix of distances between cities  
		"""
		self.nbr_villes = len( villes )
		self.villes = villes.copy()
		self.villes.insert( 0, "-------" )
		
		self.distances = np.array( distances )
		# add column of 0
		colonne = np.zeros( self.nbr_villes ).reshape( 1, self.nbr_villes )
		self.distances = np.insert( self.distances, 0, colonne, axis = 1 )
		# add line of 0
		ligne = np.zeros( self.nbr_villes+1 ).reshape( 1, self.nbr_villes+1 )
		self.distances = np.insert( self.distances, 0, ligne, axis = 0 )
		
		# find maximum distance 
		maximum = np.max( self.distances )
		
		# replace all 0 distances by maximum + 1
		self.distances[ self.distances == 0 ] = maximum + 1
	
		self.verbose = False
		self.graphics = False
		
		self.termine = False
		
		villes_dans_le_desordre = [ x for x in range(1, self.nbr_villes+1 ) ]
		random.shuffle( villes_dans_le_desordre )
		
		self.meilleure_configuration = path.Path( villes_dans_le_desordre, self. f_objectif( villes_dans_le_desordre ) )

		self.cassures = 0	
		
		
	# -----------------------------------------------------
	# Définition de la fonction objectif
	# -----------------------------------------------------
	
	def f_objectif( self, chemin ):
		"""
		Compute the total distance of a path
		"""	
		
		distance_parcourue = 0
		for i in range( 1, self.nbr_villes ):
		    distance_parcourue += self.distances[ chemin[i-1] ][ chemin[i] ]
		    
		distance_parcourue += self.distances[ chemin[ self.nbr_villes-1] ][ chemin[0] ]    
		return distance_parcourue  
    
    
    # -----------------------------------------------------
    # Perturbation du chemin courant : on choisit d'échanger
    # un certain nombre de villes
    # -----------------------------------------------------
    
	def perturbe1( self, configuration ):
	
		if self.verbeux:
			print( "\n" + "="*40 )
			print( "Perturbation" )
			print( "="*40 )
		# échanger des villes
		mini = int( self.nbr_villes * 0.1)
		maxi = int( self.nbr_villes * 0.3)
		
		nbr_de_villes_a_echanger = random.randint( mini, maxi )

		if self.verbeux:		
			print( "number of cities to modify: ", nbr_de_villes_a_echanger )
		while nbr_de_villes_a_echanger != 0:
			x = random.randint( 0, self.nbr_villes - 1 )
			y = random.randint( 0, self.nbr_villes - 1 )
			while y == x:
				y = random.randint( 0, self.nbr_villes - 1 )
				
			v1 = configuration.chemin[ x ]	
			v2 = configuration.chemin[ y ]	
			configuration.chemin[ x ] = v2
			configuration.chemin[ y ] = v1	
			nbr_de_villes_a_echanger -= 1

		configuration.distance = self.f_objectif( configuration.chemin )  
	
	
	# -----------------------------------------------------
	# Retourne la ville la plus proche n en terme de
	# distance de la ville passée en paramètre
	# -----------------------------------------------------
		 
	def ville_la_plus_proche( self, ville ):
	
		"""
		retourne la ville la plus proche de 'ville' 
		et la distance correspondante
		"""
		
		distance_ville_lpp = self.distances[ 0 ][ 0 ]
		ville_lpp = -1
		
		for i in range( 1, self.nbr_villes+1 ):
			if self.distances[ ville ][ i ] < distance_ville_lpp:
				distance_ville_lpp = self.distances[ ville ][ i ]
				ville_lpp = i
			
		return ville_lpp, distance_ville_lpp
	
	# -----------------------------------------------------
	# Perturbation du chemin courant en se basant sur la
	# distance maximum 
	# -----------------------------------------------------	    
	
	def perturbe2(self, configuration ):
	
		if self.verbeux:
			print( "\n" + "="*40 )
			print( "Perturbation" )
			print( "="*40 )

		chemin = configuration.chemin
		
		# on recherche l'indice dans le chemin de la distance
		# maximum entre deux villes
		d_max_e2v = 0
		indice_d_max_e2v = -1
		
		for i in range( 0, self.nbr_villes ):
			d = self.distances[ chemin[i-1] ][ chemin[i] ]
			if d > d_max_e2v:
				d_max_e2v = d
				indice_d_max_e2v = i-1
						
		# trouver une ville plus proche
		ville_precedente = chemin[ indice_d_max_e2v ]
		ville_suivante = chemin[ (indice_d_max_e2v + 1) % self.nbr_villes ]
		vlpp, dvlpp = self.ville_la_plus_proche( ville_precedente )
		
		chemin.remove( vlpp )
		chemin.insert( (indice_d_max_e2v + 1) % self.nbr_villes, vlpp)
		chemin.remove( ville_suivante )
		
		vlpp, dvlpp = self.ville_la_plus_proche( ville_suivante )
		i = chemin.index( vlpp )
		chemin.insert( i, ville_suivante )
		
		#print("nouveau chemin=", chemin)
		
		point_de_coupure = random.randint( 1, len(chemin)-1 )
		
		chemin = chemin[point_de_coupure:] + chemin[:point_de_coupure]
		
		configuration.distance = self.f_objectif( configuration.chemin )
		if self.verbeux:
			print("chemin = ", chemin, configuration.distance)		
		    
    
	# -----------------------------------------------------
	# Trouve les voisins améliorants à partir d'une
	# configuration
	# -----------------------------------------------------	
	
	def voisinage( self, configuration ):
		"""
		Neighborhood function that generates paths in the
		neighborhood of the current path
		
		Parameters
		----------
		configuration : path
		   current path
		"""
		
		# liste des voisins améliorants
		l = [ ]
		
		distance_a_ameliorer = configuration.distance 
		
		for i in range( 1, self.nbr_villes-2):
		    for k in range( i+1, self.nbr_villes):
		        if i > self.nbr_villes or k > self.nbr_villes:
		            continue
		        
		        nouvelle_configuration = configuration.copy()
				# échange les villes aux indices i et k
		        ville_1 = nouvelle_configuration.chemin[ i ]
		        ville_2 = nouvelle_configuration.chemin[ k ]
		        nouvelle_configuration.chemin[ i ] = ville_2
		        nouvelle_configuration.chemin[ k ] = ville_1
		        
		        nouvelle_configuration.distance = self.f_objectif( nouvelle_configuration.chemin )
		        
		        if nouvelle_configuration.distance < distance_a_ameliorer:
		            l.append( nouvelle_configuration )
		
		
		for i in range( 0, self.nbr_villes - 1 ):
			for j in range( i + 2, self.nbr_villes + 1 ):
				nouvelle_configuration = configuration.copy()
				ville = configuration.chemin[ i ]
				nouvelle_configuration.chemin.remove( ville )
				nouvelle_configuration.chemin.insert( j - 1, ville )
				nouvelle_configuration.distance = self.f_objectif( nouvelle_configuration.chemin )
				
				if nouvelle_configuration.distance < distance_a_ameliorer:
					l.append( nouvelle_configuration )

		# trier les configuration par ordre croissant de la distance        
		l.sort( key=lambda cfg : cfg.distance )
     
		return l
    
	# -----------------------------------------------------
	# Méthode de résolution basée sur la recherche locale
	# -----------------------------------------------------
	
	def _resolution( self ):
		
		"""
		Local search
		"""	
		
		# enregistre l'évolution de la distance au cours du temps
		# en fonction du nombre d'itérations
		self.liste_evolution_courante = []
		
		#
		# Génération de la configuration initiale
		#
		
		villes_dans_le_desordre = [ x for x in range(1, self.nbr_villes+1 ) ]
		random.shuffle( villes_dans_le_desordre )
		
		self.configuration_courante = path.Path( villes_dans_le_desordre, self. f_objectif( villes_dans_le_desordre ) )

		# création de la configuration courante
		self.liste_evolution_courante.append( self.configuration_courante.distance )
		
	
		self.iteration = 1
		self.cassures = 0
		
		while True:
			
			liste_voisins_ameliorants = self.voisinage( self.configuration_courante )
			
			# pas de voisin améliorant, on arrête la recherche
			if len( liste_voisins_ameliorants ) == 0:
				if (self.cassures == 4):
					break
				else:
					self.cassures += 1
					if self.cassures % 2 == 0:
						self.perturbe1( self.configuration_courante )
					else:
						self.perturbe2( self.configuration_courante )
					continue
			
			# choix du meilleur voisin améliorant
			self.configuration_courante = liste_voisins_ameliorants[ 0 ]
				
			self.iteration += 1
			
			self.liste_evolution_courante.append( self.configuration_courante.distance )

	
		return self.configuration_courante
	
	
	# -----------------------------------------------------
	# Recherche locale itérée, on garde la meilleure
	# solutions trouvée sur 'n' exécutions
	# -----------------------------------------------------
	
	def resolution( self, n ):
	
		"""
		Perform an iterated local search. We keep the best
		solution found over 'n' local searches.
		
		Parameters
		----------
		n : int
			number of time the local search is performed
		"""
		self.termine = False
		
		self.stage = 1
		
		while self.stage <= n:
			
			nouvelle_solution = self._resolution()
			
			if nouvelle_solution.distance < self.meilleure_configuration.distance:
				self.meilleure_configuration = nouvelle_solution
			
			self.stage += 1
	
		self.termine = True
		
		return self.meilleure_configuration
	
	# -----------------------------------------------------
	# Affichage du chemin ainsi que des distances 
	# parcourues et de la distance totale
	# -----------------------------------------------------	
			
	def print( self, chemin ):
	
		print()
		print( "chemin=", chemin )
		print()
		print( "------------------------------------------" )
		print( "         ville            | dist. | totale" )
		print( "------------------------------------------" )
		distance_totale = 0
		print( "{0:25s} | {1:5d} | {2:5d}".format( self.villes[ chemin[0] ], 0, 0) )
		for ville in range( 1, len( chemin ) ):	
			distance = self.distances[ chemin[ville-1] ][ chemin[ville] ]
			distance_totale += distance
			print( "{0:25s} | {1:5d} | {2:5d}".format( self.villes[ chemin[ville] ], distance, distance_totale ) )
		
		distance = self.distances[ chemin[ville] ][ chemin[0] ]
		distance_totale += distance
		print( "{0:25s} | {1:5d} | {2:5d}".format( self.villes[ chemin[0] ], distance, distance_totale ) )
		print()

	# -----------------------------------------------------
	# Affiche un graphique de l'évolution du score 
	# (distance) de la configuration courante
	# -----------------------------------------------------
	def graphique( self ):
	
		if not self.verbose:
			return 
			
		x = range( len( self.liste_evolution_courante ) )
		plt.plot( x, self.liste_evolution_courante, label='current path' )
		plt.xlabel( "iterations" )
		plt.title( "Evolution of the Local Search" )
		plt.legend()
		plt.show()


