# ###################################################################
#
#       Program: tsp_minizinc.py
#        Author: Jean-Michel Richer
#  Organisation: Computer Science Department 
#                Faculty of Science
#                University of Angers
#                2 Boulevard Lavoisier
#	             49045 Angers Cedex 01
#                France
#         Email: jean-michel.richer@univ-angers.fr
# Creation date: April, 2021
#  Modification: May, 2021
#
# ###################################################################
# 
# Aim:
#
#    This program is part of a project called TSP Visualizer
#    for Travelling Salesman Problem Visualizer that enables
#    to graphically display a solution of the TSP problem.
#    The TSPMinizinc class uses the Minizinc classes of 
#    Python to solve the problem.
#
# Objectif :
#
#    Ce programme fait partie d'un projet nommé TSP Visualizer
#    pour Visualiseur du problème du Voyageur de Commerce appelé
#	 Travelling Salesman Problem en anglais. Ce programme permet
#	 de représenter graphiquement une solution d'une instance
#	 de ce problème.
#	 La classe TSPMinizinc utilise les classes Minizinc de Python
#    afin de résoudre le problème affiché.
#
# ###################################################################
#
# License
#
#    This program is a free software you can use, modifiy and 
#    redistribute it for non profitable use. If you use this
#    code please cite the program as TSP Visualizer and the 
#	 author's name.
#    Please inform the author of possible bugs or evolution 
#    requests.
#
# Licence
#
#    Ce programme est un logiciel libre que vous pouvez utiliser, 
#    modifier et redistribuer pour un usage non lucratif. Si vous
#    utilisez ce code, merci de bien vouloir citer le nom du projet
#    TSP Visualizer et le nom de l'auteur. 
#    Merci d'informer l'auteur de bogues éventuels ou de demandes 
#    d'évolution.
#
#
# ###################################################################


import sys
import os
import re
import datetime
import subprocess
import tsp_path 
import minizinc

#
# Définition des contraintes à satisfaire ainsi que de
# l'objectif
#

minizinc_tsp = """
array[1..N] of var 1..N: x; 
int: min_val = min([distances[i,j] | i,j in 1..N where distances[i,j] > 0]);
int: max_val = max([distances[i,j] | i,j in 1..N]);
array[1..N] of var min_val..max_val: d;

constraint alldifferent(x);
constraint circuit(x);
constraint forall(i in 1..N) (
      distances[i,x[i]] = d[i]
    );

var int: distance = sum(d);
solve :: int_search(d, max_regret, indomain_split, complete) minimize distance; 
"""			


# ===================================================================
# Classe qui appelle l'exécutable minizinc
# ===================================================================

class TSPMinizinc( object ):

	# =====================================================
	# Constructeur
	# =====================================================
	
	def __init__( self, fichier, distances ):
	
		"""
		WHAT	
			Constructor with file and matrix of distances
		
		HOW
			Create a TSP instance with the matrix of distances
			provided and initialize the model 
			
		PARAMETERS
			- fichier (string) : name of file
			- distances (matrix of numbers) : matrix of distances
		
		"""
		
		self.termine = False
		
		self.fichier = fichier
		self.distances = distances
		self.minizinc_binary = None
		self.nbr_villes = len( distances )
		self.meilleure_configuration = None
		self.model = minizinc.Model()
		self.nom_solveur = "chuffed"
		self.temps_maximum = 60
		try:
			s = "include \"globals.mzn\";\n" 
			s += "int: N = " + str( len( distances ) ) + ";\n" 
			s += "array[1..N, 1..N] of int: distances;\n" 
			s += "distances = array2d(1..N,1..N,\n" 
			s += "[\n" 
			for y in range( len( distances ) ):
				tmp = ""
				for x in range( len( distances ) ):
					tmp += str( int( distances[ y ][ x ] ) ) + ", "
				s += tmp + "\n"
			s += "]);\n" 
			s += "\n" 
			s += minizinc_tsp
			
			#print( s )
			self.model.add_string( s )
			
		except:	
			# in cas of an exception return the exception message
			print( "="*50 )
			print( "ERROR : tsp_minizinc.__init__()" )
			print( "="*50 )
			#print( err)
			print( "="*50 )
			sys.exit( 1 )
	
	# -----------------------------------------------------
	# Définition des paramètres du solveur qui sont
	# le nom du solveur ou le temps de résolution
	# maximum accordé pour la résolution
	# -----------------------------------------------------	
	def definition_parametres( self, minizinc ):
	
		"""
		WHAT
			Definition of the parameters of the solver which
			are the name of the solver and the time dedicated
			to the search of a solution
			
		PARAMETERS
			- minizinc (array of string) : first element of the
			is the name of the solver or the empty string, the 
			second element is the time dedicated to the search
			as a string or an empty string
		"""
		
		a = minizinc.split( "," )
		
		if len( a ) >= 1:
			if len( a[0] ) != 0:
				self.nom_solveur = a[0]
		if len( a ) >= 2:
			if len( a[1] ) != 0:
				self.temps_maximum = int( a[1] )
			
		
	# -----------------------------------------------------
	# Définition de la fonction objectif
	# -----------------------------------------------------
	
	def f_objectif( self, chemin ):
	
		"""
		WHAT
			Objective (or score) function that computes
			the distance of the path
			
		HOW
			We add the distances from one city to the next city
			
		PARAMETERS
			- chemin (list of int): a path
			
		RETURN
			an float value	
		"""	
		
		distance_parcourue = 0
		for i in range( 1, self.nbr_villes ):
		    distance_parcourue += self.distances[ chemin[i-1] ][ chemin[i] ]
		    
		distance_parcourue += self.distances[ chemin[ self.nbr_villes-1] ][ chemin[0] ]    
		return distance_parcourue
			
		
	# -----------------------------------------------------
	# Résolution du problème du Voyageur de Commerce
	# -----------------------------------------------------			
	def resolution( self ):
	
		"""
		WHAT
			Resolution of the problem with the Python Minizinc
			classes
			
		HOW
			We create the solver and the instance and set the 
			timeout. When the resolution is finished we get
			the path and return the solution
			
		RETURN
			a tsp_path.Path that contains the path and the 
			distance of the path
		"""
		
		self.termine = False
		
		try:
			# use chuffed solver
			solver = minizinc.Solver.lookup( self.nom_solveur )
			instance = minizinc.Instance( solver, self.model )
			
			# allow search for 1 minute
			timeout = datetime.timedelta( seconds = self.temps_maximum )
			self.result = instance.solve( intermediate_solutions = True, timeout=timeout )

			distance_minimum = self.result[0].objective
			chemin_minimum = self.result[0].x
			for i in range(1, len ( self.result ) ):
				if self.result[i].objective < distance_minimum:
					distance_minimum = self.result[i].objective
					chemin_minimum =  self.result[i].x
					
			# create path
			l = { }

			for src in range(0, len(chemin_minimum)):
				dst = chemin_minimum[ src ]-1
				d = self.distances[ src ][ dst ]

				l[src] = [dst , d]			

			start = 0
			villes = [ start ]
			next = l[ start ][ 0 ]

			while next != 0:
				start = next
				villes.append( start )
				next = l[ start ][ 0 ]
			
		except:	
			# in cas of an exception return the exception message
			print( "="*50 )
			print( "ERROR : tsp_minizinc.resolution()" )
			print( "="*50 )
			print( sys.exc_info()[0] )
			print( sys.exc_info()[1] )
			print( "="*50 )
			villes = [x for x in range( len( self.distances ) )]


		villes_p1 = [ x+1 for x in villes ]
		self.meilleure_configuration = tsp_path.Path( villes_p1 , self.f_objectif( villes ) )
			
		self.termine = True
		
		return self.meilleure_configuration
	
	# -----------------------------------------------------
	# Il n'y a pas de graphique à afficher
	# -----------------------------------------------------		
	
	def graphique( self ):
		pass
		

