# ###################################################################
#
#       Program: tsp_lkh.py
#        Author: Jean-Michel Richer
#  Organisation: Computer Science Department 
#                Faculty of Science
#                University of Angers
#                2 Boulevard Lavoisier
#	             49045 Angers Cedex 01
#                France
#         Email: jean-michel.richer@univ-angers.fr
# Creation date: April, 2021
#  Modification: May, 2021
#
# ###################################################################
# 
# Aim:
#
#    This program is part of a project called TSP Visualizer
#    for Travelling Salesman Problem Visualizer that enables
#    to graphically display a solution of the TSP problem.
#    The TSPLKH class calls the lkh binary to solve the 
#	 problem that is displayed
#
# Objectif :
#
#    Ce programme fait partie d'un projet nommé TSP Visualizer
#    pour Visualiseur du problème du Voyageur de Commerce appelé
#	 Travelling Salesman Problem en anglais. Ce programme permet
#	 de représenter graphiquement une solution d'une instance
#	 de ce problème.
#	 La classe TSPLKH appelle l'exécutable lkh afin de résoudre
#    le problème affiché.
#
# ###################################################################
#
# License
#
#    This program is a free software you can use, modifiy and 
#    redistribute it for non profitable use. If you use this
#    code please cite the program as TSP Visualizer and the 
#	 author's name.
#    Please inform the author of possible bugs or evolution 
#    requests.
#
# Licence
#
#    Ce programme est un logiciel libre que vous pouvez utiliser, 
#    modifier et redistribuer pour un usage non lucratif. Si vous
#    utilisez ce code, merci de bien vouloir citer le nom du projet
#    TSP Visualizer et le nom de l'auteur. 
#    Merci d'informer l'auteur de bogues éventuels ou de demandes 
#    d'évolution.
#
#
# ###################################################################


import sys
import os
import re
import subprocess
import tsp_path 

# ===================================================================
# Classe qui appelle l'exécutable lkh
# ===================================================================

class TSPLKH(object):

	# =====================================================
	# Constructeur
	# =====================================================
	
	def __init__( self, fichier, distances ):
		self.termine = False
		
		self.fichier = fichier
		self.distances = distances
		self.lkh_binary = None
		self.nbr_villes = len( distances )
		self.meilleure_configuration = None
		
		try:
			# créer un répertoire temporaire
			repertoire_temoraire = "tmp"
			if not os.path.isdir( repertoire_temoraire ):
				os.mkdir( repertoire_temoraire )
			
			# !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
			# modification du nom du fichier pour supprimer
			# les répertoires dans le nom
			# !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
			fichier = os.path.basename( fichier )
				
			# créer un fichier tsp
			
			nom_fichier_tsp = repertoire_temoraire + os.path.sep + fichier + ".tsp"
			file = open( nom_fichier_tsp, "w+" )
			file.write( "NAME: " + fichier + "\n" )
			file.write( "TYPE: TSP\n" )
			file.write( "DIMENSION: " + str( len( distances ) ) + "\n" )
			file.write( "EDGE_WEIGHT_TYPE: EXPLICIT\n" )
			file.write( "EDGE_WEIGHT_FORMAT: FULL_MATRIX\n" ) 
			file.write( "EDGE_WEIGHT_SECTION\n" )
			for y in range( len( distances ) ):
				s = ""
				for x in range( len( distances ) ):
					s += str( int( distances[ y ][ x ] ) ) + " "
				file.write( s + "\n" )
			file.write( "EOF\n" )
			file.close()
			
		
			# créer un fichier .par pour lkh
			self.fichier_par = repertoire_temoraire + os.path.sep + fichier + ".par"
			file = open( self.fichier_par, "w+" )
			file.write( "PROBLEM_FILE = " + nom_fichier_tsp + "\n" )
			file.write( "RUNS = 10\n")
			self.fichier_tour = repertoire_temoraire + os.path.sep + fichier + "_tour.txt"
			file.write( "OUTPUT_TOUR_FILE=" + self.fichier_tour + "\n" )
			
		except (RuntimeError, IOError)  as err:	
			# in cas of an exception return the exception message
			print( "="*50 )
			print( "ERROR : tsp_lkh.__init__()" )
			print( "FILE  : '" + self.fichier + "'" )
			print( "="*50 )
			print( err)
			print( "="*50 )
			sys.exit( 1 )
	
	# -----------------------------------------------------
	#
	# -----------------------------------------------------	
	def definition_programme( self, lkh ):
		self.lkh_binary = lkh
		
	# -----------------------------------------------------
	# Définition de la fonction objectif
	# -----------------------------------------------------
	def f_objectif( self, chemin ):
		"""
		Compute the total distance of a path
		"""	
		
		distance_parcourue = 0
		for i in range( 1, self.nbr_villes ):
		    distance_parcourue += self.distances[ chemin[i-1] ][ chemin[i] ]
		    
		distance_parcourue += self.distances[ chemin[ self.nbr_villes-1] ][ chemin[0] ]    
		return distance_parcourue
			
	# -----------------------------------------------------
	# Récupère le chemin
	# -----------------------------------------------------
			
	def finalisation( self, s ):
		villes = s.split( '\n' )
		if len(villes) != self.nbr_villes:
			raise RuntimeError("Found " + str( len( villes ) ) + " cities in PATH section\n" + "while " + str( self.nbr_villes ) + " are expected\n" )
		villes = [ int(x)-1 for x in villes ]
				
		villes_p1 = [ x+1 for x in villes ]
		
		self.meilleure_configuration = tsp_path.Path( villes_p1, self.f_objectif( villes ) )
		
		
	# -----------------------------------------------------
	# -----------------------------------------------------			
	def resolution( self ):
		self.termine = False
		
		result = subprocess.run( [ self.lkh_binary, self.fichier_par ] )
			
		try:
			file = open( self.fichier_tour, "r" )
			contents = file.read()
			file.close()
			
			match = re.search( "TOUR_SECTION[\s]*([^\]]+)-1", contents )
			if match:
				self.finalisation( match.group(1).strip() )	
				
		except (RuntimeError, IOError)  as err:	
			# in cas of an exception return the exception message
			print( "="*50 )
			print( "ERROR : tsp_lkh.resolution()" )
			print( "FILE  : '" + self.fichier + "'" )
			print( "="*50 )
			print( err)
			print( "="*50 )
			sys.exit( 1 )
		self.termine = True
		
		return self.meilleure_configuration
	
	# -----------------------------------------------------
	# Il n'y a pas de graphique à afficher
	# -----------------------------------------------------		
	
	def graphique( self ):
		pass
		

