# ###################################################################
#
#       Program: solveur.py
#        Author: Jean-Michel Richer
#  Organisation: Computer Science Department, 
#                University of Angers,
#	             France
#         Email: jean-michel.richer@univ-angers.fr
# Creation date: April, 2021
#  Modification: May, 2021
#
# ###################################################################
# 
# Aim:
#
#    This program is a general class that is bind to a solver
#    of the TSP
#
# Objectif :
#
#    Ce programme est une classe générale liée à la résolution
#    du problème du Voyageur de Commerce.
#
# ###################################################################
#
# License
#
#    This program is a free software you can use, modifiy and 
#    redistribute it for non profitable use. If you use this
#    code please cite the program as TSP Visualizer and the 
#	 author's name.
#    Please inform the author of possible bugs or evolution 
#    requests.
#
# Licence
#
#    Ce programme est un logiciel libre que vous pouvez utiliser, 
#    modifier et redistribuer pour un usage non lucratif. Si vous
#    utilisez ce code, merci de bien vouloir citer le nom du projet
#    TSP Visualizer et le nom de l'auteur. 
#    Merci d'informer l'auteur de bogues éventuels ou de demandes 
#    d'évolution.
#
# ###################################################################

import sys
import tsp_recuit_simule as trs
import tsp_recherche_locale as trl
import tsp_lkh as trk
import tsp_minizinc as trm
import eztsp_loader as tl

# ===================================================================
# Création d'un Solveur qui est une classe qui permet de résoudre
# le problème en utilisant l'une des méthodes de résolution 
# disponible, à savoir :
# - un algorithme de recherche locale itérée, rapide mais qui ne
#   donne généralement pas de solution de bonne qualité
# - un algorithme de recuit simulé, très lent mais qui permet parfois
#   d'atteindre des solutions de très bonne qualité
# ===================================================================

class Solveur( object ):

	"""
	Definition of a Solver, i.e. a class that enables to solve the
	Trasvelling Salesman Problem using either:
	- an iterated local search algorithm that is fast but generally
	  provides solutions of poor quality
	- a simulated annealing algorithm that is slow bu can prodivde
	  solution of very good quality
	"""
	
	# =====================================================
	# Constructeur avec méthode de résolution et fichier
	# =====================================================
	
	def __init__( self, methode, fichier, param ):
	
		"""
		Constructor given two parameters
		
		Parameters
		----------
		- methode: string
			the method used, either "Descente" for a local search or
			"Recuit" for the simulated annealing algorithm
		- ficher: string
			file that contains the definition of the problem to solve
		-param : string
			either lkh binary or Minizinc parameters
		"""
		
		self.fichier = fichier
		self.methode = methode
		self.loader = tl.EZTSPLoader( fichier )
		self.loader.load()
		if methode == "Descente":
			self.tsp = trl.TSPRechercheLocale( self.loader.list_of_cities, self.loader.matrix_of_distances )	
		elif methode == "Recuit":
			self.tsp = trs.TSPRecuitSimule( self.loader.list_of_cities, self.loader.matrix_of_distances )	
		elif methode == "LKH":
			self.tsp = trk.TSPLKH( self.fichier, self.loader.matrix_of_distances )	
			self.tsp.definition_programme( param )
		elif methode == "Minizinc":
			self.tsp = trm.TSPMinizinc( self.fichier, self.loader.matrix_of_distances )	
			self.tsp.definition_parametres( param )
		
		self.tsp.verbeux = False
	
	# =====================================================
	# Résolution du problème avec les paramètres adéquates
	# =====================================================
		
	def resoudre( self ):
		
		"""
		Resolution of the problem with the appropriate parameters
		- for an iterated local search: 10 iterations
		- for a simulated annealing algorithm:
		  -- initial temperature 1000.0
		  -- final temperature 0.01
		  -- cooling factor of 0.999
		"""
		
		if self.methode == "Descente":
			self.configuration_finale = self.tsp.resolution( 10 )	
			
		elif self.methode == "Recuit":
			self.configuration_finale = self.tsp.resolution( 1000.0, 0.01, 0.999 )
			
		elif self.methode == "LKH":
			self.configuration_finale = self.tsp.resolution()
			
		elif self.methode == "Minizinc":
			self.configuration_finale = self.tsp.resolution()

