# ###################################################################
#
#       Program: tsp_loader.py
#        Author: Jean-Michel Richer
#  Organisation: Computer Science Department 
#                Faculty of Science
#                University of Angers
#                2 Boulevard Lavoisier
#	             49045 Angers Cedex 01
#                France
#         Email: jean-michel.richer@univ-angers.fr
# Creation date: April, 2021
#  Modification: May, 2021
#
# ###################################################################
# 
# Aim:
#
#    This program is part of a project called TSP Visualizer
#    for Travelling Salesman Problem Visualizer that enables
#    to graphically display a solution of the TSP problem.
#    The EZTSPLoader class is used to read a file in EZTSP
#	 format (see the README.md file)
#
# Objectif :
#
#    Ce programme fait partie d'un projet nommé TSP Visualizer
#    pour Visualiseur du problème du Voyageur de Commerce appelé
#	 Travelling Salesman Problem en anglais. Ce programme permet
#	 de représenter graphiquement une solution d'une instance
#	 de ce problème.
#	 La class EZTSPLoader est utilisée afin de lire un fichier au
#    format EZTSP (voir le fichier README.md)
#
# ###################################################################
#
# License
#
#    This program is a free software you can use, modifiy and 
#    redistribute it for non profitable use. If you use this
#    code please cite the program as TSP Visualizer and the 
#	 author's name.
#    Please inform the author of possible bugs or evolution 
#    requests.
#
# Licence
#
#    Ce programme est un logiciel libre que vous pouvez utiliser, 
#    modifier et redistribuer pour un usage non lucratif. Si vous
#    utilisez ce code, merci de bien vouloir citer le nom du projet
#    TSP Visualizer et le nom de l'auteur. 
#    Merci d'informer l'auteur de bogues éventuels ou de demandes 
#    d'évolution.
#
#
# ###################################################################

import re
import sys
import os

# ===================================================================
# Classe utilisée pour charger les données contenues dans un ficher
# d'extension .eztsp
# ===================================================================

class EZTSPLoader(object):

	"""
	Class used to load data from a file with .tsp extension
	"""
	
	# =====================================================
	# Constructeur
	# =====================================================
	
	def __init__(self, file_name):
		self.file_name = file_name
		# number of cities
		self.count = 0
		self.list_of_cities = []
		self.matrix_of_distances = []
		self.path = []
		self.window_dimensions = [ 800, 600 ]
		self.list_of_positions = []
		self.background = ""
	
	# =====================================================
	# Analyse des données liées au champ COUNT qui indique
	# le nombre de villes présentes
	# =====================================================
		
	def section_count(self, s):
	
		"""
		Analysis of data that correspond to the COUNT field
		which records the number of cities
		"""
		
		self.count = int(s)
		if self.count < 3 or self.count > 150:
			raise RuntimeError("In section COUNT, the application can only display between 3 and 150 cities")
	

	# =====================================================
	# Analyse des données liées au champ DISTANCES qui 
	# donne les distances deux à deux entre villes
	# =====================================================
	
	def section_distances(self, s):

		a = s.split('\n')
		
		nbr_lines =0
		
		for i in range(len(a)):
			t = a[i].strip()
			if len(t) == 0:
				continue
			b = t.split(",")

			l = []
			for j in range(len(b)):
				value = float( b[j].strip() )
				if i == j:
					if value != 0:
						raise RuntimeError("Diagonal value should be 0")
				l.append( value )					

			if len(l) != self.count:
				raise RuntimeError("Found " + str(len(l)) + " cities in DISTANCES section\n" + "while " + str(self.count) + " are expected\n" + "on line: " + s)
					
			self.matrix_of_distances.append( l )
			nbr_lines += 1
			
		if nbr_lines != self.count:
			raise RuntimeError("Found " + str(nbr_lines) + " lines in DISTANCES section\n" + "while " + str(self.count) + " are expected")
			
			
	# =====================================================
	# Analyse des données liées au champ CITIES qui 
	# donne les noms des villes
	# =====================================================

	def section_cities(self, s):
		a = s.split('\n')
		for i in range(len(a)):
			a[i] = a[i].strip()
		self.list_of_cities = a
		if len(a) != self.count:
			raise RuntimeError("Found " + str(len(a)) + " names in section NAMES\n" + "while " + str(self.count) + " are expected")
			
	# =====================================================
	# Analyse des données liées au champ PATH qui 
	# donne le chemin à afficher
	# =====================================================

	def section_path(self, s):
		a = s.split(',')
		if len(a) != self.count:
			raise RuntimeError("Found " + str(len(a)) + " cities in PATH section\n" + "while " + str(self.count) + " are expected\n" + "on line: " + s)
		for i in range(len(a)):
			city_id = int(a[i].strip())
			if city_id < 1 or city_id > self.count:
				raise RuntimeError("City identifier can only range from 1 to " + str(self.count) + " in PATH section" )
			if city_id in self.path:
				raise RuntimeError("City of id " + str(city) + " appears twice in PATH section" )
			self.path.append( city_id )
			
			
	# =====================================================
	# Analyse des données liées au champ WINDOW qui 
	# donne les dimensions de la fenêtre d'affichage
	# =====================================================

	def section_window(self, s):
		a = s.split(',')
		if len(a) != 2:
			raise RuntimeError("Found " + str(len(a)) + " values in WINDOW section\n" + "while exactly two are expected\n" + "on line: " + s)
		x = int( a[0] )
		y = int( a[1] )
	
		if x <= 320 or x >= 1920:
			raise RuntimeError("Dimension for width of window should be between 320 and 1920")
		if y <= 200 or y >= 1080:
			raise RuntimeError("Dimension for height of window should be between 320 and 1920")
			
		self.window_dimensions = [ x, y ]
	
	# =====================================================
	# Analyse des données liées au champ POSITION qui 
	# donne les coordonnées des villes, celle-ci seront
	# modifiées par la suite pour être affichées dans la 
	# fenêtre graphique
	# =====================================================
	
	def section_positions(self, s):
		a = s.split('\n')
		nbr_lines = 0
		for i in range(len(a)):
			t = a[i].strip()
			if len(t) == 0:
				continue
			b = t.split(",")
			if len(b) != 2:
				raise RuntimeError("Only two coordinates are expected in section POSITIONS\n" + "in line " + s)
			l = [ float(b[0]), float(b[1]) ]
			self.list_of_positions.append( l )
			nbr_lines += 1	
			
		if nbr_lines != self.count:
			raise RuntimeError("Found " + str(nbr_lines) + " lines in section POSITIONS\n" + "while " + str(self.count) + " are expected")
			
	def section_background(self, s):
		if not os.path.isfile(s):
			raise RuntimeError("Background file " + s + " does not exist")
		self.background = s	
			
	# =====================================================
	# Lecture des différentes sections qui compose le 
	# fichier de définition de données. Seules les
	# sections CITIES et DISTANCES sont obligatoires.
	# Les sections DISTANCES, PATH, WINDOW, POSITIONS
	# sont optionnelles
	# =====================================================
			
	def load(self):
		"""
		Read the sections that compose the file of data which
		should have a .tsp extension.
		There are two sections required:
		- COUNT which defines the number of cities
		- DISTANCES which defines the distance matrix
		There are four optional sections:
		- NAMES which defines the names of the cities
		- PATH which defines the path to display
		- POSITION which defines the positions of the cities
		- WINDOW which defines the dimension of the window
		  used to display the cities and path
		  
		  Return value
		  ------------
		  The return value is either an error message or an
		  empty string if no error was found
		"""
		try:
			#
			# read entire file and then look for sections
			#
			
			file = open(self.file_name)
			contents = file.read()
			file.close()

			#
			# required sections
			#
						
			match = re.search("COUNT[\s]*\[([^\]]+)\]", contents)
			if not match:
				raise RuntimeError("COUNT section not found")
			else:
				self.section_count( match.group(1).strip() )	

			match = re.search("DISTANCES[\s]*\[([^\]]+)\]", contents)
			if not match:
				raise RuntimeError("DISTANCES section not found")
			else:
				self.section_distances( match.group(1).strip() )	

			#
			# optional sections
			#
			
			match = re.search("CITIES[\s]*\[([^\]]+)\]", contents)
			if match:
				self.section_cities( match.group(1).strip() )	

			match = re.search("PATH[\s]*\[([^\]]+)\]", contents)
			if match:
				self.section_path( match.group(1).strip() )	
			
			match = re.search("WINDOW[\s]*\[([^\]]+)\]", contents)
			if match:
				self.section_window( match.group(1).strip() )	
			
			match = re.search("POSITIONS[\s]*\[([^\]]+)\]", contents)
			if match:
				self.section_positions( match.group(1).strip() )	

			match = re.search("BACKGROUND[\s]*\[([^\]]+)\]", contents)
			if match:
				self.section_background( match.group(1).strip() )	
			
			
		except (RuntimeError, IOError)  as err:	
			# in cas of an exception return the exception message
			print( "="*50 )
			print( "ERROR : eztsp_loader.load()" )
			print( "FILE  : '" + self.file_name + "'" )
			print( "="*50 )
			print( err)
			print( "="*50 )
			sys.exit( 1 )
			

		# no error, return empty string		
		return ""
		
	
	
		
