# ###################################################################
#
#       Program: ez_pygame_fonts_manager.py
#        Author: Jean-Michel Richer
#  Organisation: Computer Science Department 
#                Faculty of Science
#                University of Angers
#                2 Boulevard Lavoisier
#	             49045 Angers Cedex 01
#                France
#         Email: jean-michel.richer@univ-angers.fr
# Creation date: May, 2021
#  Modification: May, 2021
#
# ###################################################################
# 
# Aim:
#
#    This program is a small module that helps store fonts
#    that will be used to draw text with pygame. It is part
#	 of the ez_pygame library.
#
# Objectif :
#
#    Ce programme est un petit module qui permet de stocker les
#    polyces de caractères utilisées pour dessiner du texte avec
#    pygame. Ce programme fait partie de la librairie ez_pygame.
#
# ###################################################################
#
# License
#
#    This program is a free software you can use, modifiy and 
#    redistribute it for non profitable use. If you use this
#    code please cite the library (ez_pygame) and author.
#    Please inform the author of possible bugs or evolution 
#    requests.
#
# Licence
#
#    Ce programme est un logiciel libre que vous pouvez utiliser, 
#    modifier et redistribuer pour un usage non lucratif. Si vous
#    utilisez ce code, merci de bien vouloir citer le nom de la
#    librairie (ez_pygame) et l'auteur. 
#    Merci d'informer l'auteur de bogues éventuels ou de demandes 
#    d'évolution.
#
#
# ###################################################################

import pygame
from pygame.locals import *

# ===================================================================
# Class principale qui stocke les polices de caractères et permet 
# d'y faire référence par un identifiant sous forme de chaîne de
# caractères
# ===================================================================

class FontsManager( object ):

	# =====================================================
	# Constructeur qui permet de définir les fontes à 
	# stocker en fournissant des informations sur le fichier
	# TTF (True Type Font) ainsi que les tailles de polices
	# =====================================================

	def __init__( self, fonts_def ):
	
		"""
		WHAT
			Constructor that will record the fonts and their
			sizes. Note that a default font is defined as
			Ubuntu Regular in 14 pts. This font will be used
			if the user is requesting a font that is not 
			recorded.
		
		PARAMETERS
			- fonts_def (dict): parameter that defines the fonts
			
		USAGE
		```
		import ez_pygame_fonts_manager as fm
		fonts_definition = {
		
			"pm" : { "ttf_file": "fonts/PermanentMarker-Regular.ttf", 
					 "sizes": [10,12,14,16,18,20,22,24, 32] },
			"um" : { "ttf_file": "fonts/UbuntuMono-R.ttf",  
					 "sizes": [16,18,20,22,24] }
		}
		fma = fm.FontsManager( fonts_definition )
		# set the font Ubuntu Mono (Regular) in 16 points
		# as the current font
		fma.set_current_font( "um16" )
		```
		"""
		
		self.list_fonts = {}

		self.default_font = pygame.font.Font( "fonts/Ubuntu-R.ttf", 14 )
		self.current_font = self.default_font
		self.list_fonts[ "default" ] = self.current_font
		
		for font_name, properties in fonts_def.items():
			for size in properties[ "sizes" ]:
				font_full_name = font_name + str( size )
				font = pygame.font.Font( properties[ "ttf_file" ], size )
				self.list_fonts[ font_full_name ] = font
				
	# =====================================================
	# Definit la police qui serait utilisée pour afficher
	# du texte 
	# =====================================================
	
	def set_current_font( self, name ):
	
		"""
		WHAT
			This subprogram defines the current font given its
			identifier which must be the short name of the font
			(for example um for Ubuntu Mono) as defined by the
			user in the 'fonts_def' parameter of the constructor
			followed ny the size in points.
		
		PARAMETERS
			- name (string): name of font (for example "um16" for 
			Ubuntu Mono in 16 points) as defined by the user
		"""
		
		if name in self.list_fonts:
			self.current_font = self.list_fonts[ name ]
			return self.list_fonts[ name ]
		else:
			self.current_font = self.default_font
			return self.default_font

	# =====================================================
	# Retourne la fonte courante définie par le sous-programme
	# précédent
	# =====================================================
	
	def get_current_font( self ):
	
		"""
		WHAT
			Get current font
			
			
		"""
		return self.current_font
		
		
	# =====================================================
	# Retourne la police demandée identifiée par son nom
	# sous forme de chaine de caractère
	# =====================================================	
	
	def get_font( self, name ):
	
		"""
		WHAT	
			Return the font given its identifier
		
		PARAMETERS
			- name (string): name of font (for example "um16" for 
			Ubuntu Mono in 16 points) as defined by the user
		"""
		
		if not name	in self.list_fonts:
			return self.default_font
		
		return self.list_fonts[ name ]


		

