# ###################################################################
#
#       Program: ez_pygame.py
#        Author: Jean-Michel Richer
#  Organisation: Computer Science Department 
#                Faculty of Science
#                University of Angers
#                2 Boulevard Lavoisier
#	             49045 Angers Cedex 01
#                France
#         Email: jean-michel.richer@univ-angers.fr
# Creation date: April, 2021
#  Modification: April, 2021
#
# ###################################################################
# 
# Aim:
#
#    This program is a small layer that facilitates the use of the
#    pygame library. It enables to define the window title and 
#    dimensions and handles the fonts and the display of text.
#    The main function is the 'render' function that must be
#    called with two user defined function for rendering and
#    events handling
#
# Objectif :
#
#    Ce programme représente une petite surcouche à la librairie
#    pygame et permet de définit le titre de la fenêtre ainsi que
#    ses dimensions. On essaye également de faciliter l'utilisation
#    des polices de caractères et l'affichage de texte dans la fenêtre.
#    La fonction principale appelée 'render' doit être appelée avec
#    deux fonctions définies par l'utilisateur chargées de l'affichage
#    et de la gestion des évènements.
#
# ###################################################################
#
# License
#
#    This program is a free software you can use, modifiy and 
#    redistribute it for non profitable use. If you use this
#    code please cite the library (ez_pygame) and author.
#    Please inform the author of possible bugs or evolution 
#    requests.
#
# Licence
#
#    Ce programme est un logiciel libre que vous pouvez utiliser, 
#    modifier et redistribuer pour un usage non lucratif. Si vous
#    utilisez ce code, merci de bien vouloir citer le nom de la
#    librairie (ez_pygame) et l'auteur. 
#    Merci d'informer l'auteur de bogues éventuels ou de demandes 
#    d'évolution.
#
#
# ###################################################################

import pygame
from pygame.locals import *
import sys
import fonts_manager

# ===================================================================
# Classe principale qui agit comme une surcouche de pygame.
# ===================================================================

class EZPygame( object ):

	# =====================================================
	# Constructeur qui permet de définir le titre de la 
	# fenêtre ainsi que ses dimensions et la couleur de
	# fond
	# =====================================================
	
	def __init__( self, title, window_width = 640, window_height = 480, bg = (255,255,255) ):
	
		"""
		WHAT
			Constructor that enables to define the title and dimensions 
			of the window as well as the background color
		
		PARAMETERS
			- title (string): title of the window
			- window_width (int): width of the window in pixels
			- window_height (int): height of the window in pixels
			- bg (x,y,z): background color as RGB
			- fm (ez_pygame_fonts_manager.FontManager): manager in charge
				of fonts and their sizes
		"""
		
		# initialize pygame and set 
		pygame.init()
		self.window_width = window_width
		self.window_height = window_height
		self.window = pygame.display.set_mode( [window_width, window_height] )
		self.background_color = bg
		pygame.display.set_caption( title )
		pygame.font.init()
		self.default_font = None
		self.separator = "="*60
		# Font manager
		self.fma = None
		
	# =====================================================
	# Définition de la fonte courante
	# =====================================================
		
	def set_font_manager( self, fma ):
	
		"""
		WHAT
			Define the class in charge of fonts
			
		PARAMETERS
			- fma (FontManager)
		"""
		
		self.fma = fma
		
	
	# =====================================================
	# Obtenir une fonte en fonction de son identifiant
	# =====================================================		
		
	def get_font( self, name ):
		"""
		WHAT
			Get font given its identifier as a string

		HOW
			We make use of the FontManager (self.fma)
				
		PARAMETERS
			- name (string): identifier of font	
			
		"""
		
		return self.fma.get_font( name )

	# =====================================================
	# Afficher du texte susceptible de contenir des retour
	# à la ligne.
	# On doit spécifier la chaine à afficher, sa position
	# sur l'écran ainsi que la couleur d'affichage.
	# Les autres paramètres sont optionnels :
	# - si la couleur de fond n'est pas précisée c'est
	#   celle qui a été définie dans le constructeur qui
	#   est utilisée
	# - La fonte d'affichage peut être définie soit en la
	#   passant à 'font' ou en précisant son nom 'font_name'.
	#   Si elle n'est pas précisée c'est la fonte par défaut
	#   qui est utilisée
	# - si 'center' est positionnée à True, alors la position
	#   (x,y) est utilisée comme centre du texte
	# =====================================================		
		
	def draw_text( self, s, x, y, fg_color, bg_color=None, font=None, font_name="", center = False ):
	
		"""
		WHAT
			Draw text (string with carriage return) inside window 
			at a given position. If the text contains '\n' caracters 
			we pass to the next line.
			The required parameters are the text to display, its
			coordinates in the window and a foreground color.
			Optional parameters are:
			- the background color, that will be the background color
			provided in the constructor if not provided
			- the font to use which can be defined either using the
			'font' parameter and or the name of the font (parameter
			'font_name')
			- if center is positioned to True then the position (x,y)
			provided is used as the center of the text

		PARAMETERS
		- s (string): the text to draw which can contain carriage return.
			When '\n' is found we pass to the next line and
			modify the y position of the next 
		- x (int): coordinate on the x-axis in window
		- y (int): coordinate on the y-axis in window
		- fg_color (R,G,B): foreground color as a RGB triplet
		- bg_color (R,G,B): background color as a RGB triplet, if not 
			specified (None) then we use the background color defined 
			in the constructor
		- font (pygame.font): font to use
		- font_name (string): identifier of the font to use defined in
			the FontManager, so use either font or font_name but not both
		- center (bool): if True the position (x,y) is the center of the text	
		
		RETURN
			We return (y) the coordonate of the last line of text and
		 	(rect) the position and dimension of the last line
		"""
		
		# check parameters
		if font == None:
			font = self.fma.get_current_font()
		if len(font_name) == 0:
			font = self.fma.get_current_font()
		else:
			font = self.fma.get_font( font_name )
			
		if bg_color == None:
			bg_color = self.background_color
		
		# compute height of characters if we need to pass to the
		# next line when 's' contains '\n'
		render = font.render( "Tg", True, fg_color )
		rect = render.get_rect() 
		font_height = rect[3]
				
		array = s.split( "\n" )
		# for each line of text
		for s in array:
			# print text or pass to next line if line is empty
			if len(s) > 0:
				render = font.render( s, True, fg_color )
				rect = render.get_rect() 
				if center == False:
					rect.move_ip( x, y )
				else:
					rect.center = (x, y)	
				pygame.draw.rect( self.window, bg_color, rect )
				self.window.blit( render, rect )
			y += font_height
			
		return y, rect	
	
	# =====================================================
	# Affiche un texte au centre de l'écran
	# =====================================================

	def draw_text_center( self, s, y, fg_color, bg_color=None, font=None, font_name="" ):
	
		"""
		Display text (string with carriage return) at the
		center of the window (See method 'draw_text' for further
		explanations).
		
		PARAMETERS
		- s (string): the text to draw which can contain carriage return.
			When '\n' is found we pass to the next line and
			modify the y position of the next 
		- x (int): coordinate on the x-axis in window
		- y (int): coordinate on the y-axis in window
		- fg_color (R,G,B): foreground color as a RGB triplet
		- bg_color (R,G,B): background color as a RGB triplet, if not 
			specified (None) then we use the background color defined 
			in the constructor
		- font (pygame.font): font to use
		- font_name (string): identifier of the font to use defined in
			the FontManager, so use either font or font_name but not both
		- center (bool): if True the position (x,y) is the center of the text	
		
		RETURN
			We return (y) the coordonate of the last line of text 
		"""
		
		if font == None:
			font = self.fma.get_current_font()
		if len(font_name) == 0:
			font = self.fma.get_current_font()
		else:
			font = self.fma.get_font( font_name )
			
		if bg_color == None:
			bg_color = self.background_color
		
		render = font.render( "T", True, fg_color )
		rect = render.get_rect() 
		font_height = rect[3]
				
		array = s.split("\n")
		for s in array:
			if len(s) > 0:
				render = font.render( s, True, fg_color )
				rect = render.get_rect() 
				rect.center = ( self.window_width // 2, y )
				pygame.draw.rect( self.window, bg_color, rect )
				self.window.blit( render, rect )
			y += font_height
	
		return y
		
		
	# =====================================================
	# Gestionnaire d'évènements par défaut
	# =====================================================	
	
	def default_events_handler( self, gui ):
	
		"""
		WHAT
			Default events handler if not specified in 'render()'
			If quit has been activated (Ctrl-C, F10 or Alt+F4) then 
			set variable gui.stop to True to exit from rendering loop.
			
		PARAMETERS
			- gui (pygame.Surface): window 	
		"""
		
		for event in pygame.event.get():
			if event.type == QUIT:
				gui.stop = True	

	# =====================================================
	# Fonction de rendu par défaut
	# =====================================================	
				
	def default_rendering_function( self, gui ):
	
		"""
		WHAT
			Default rendering function if not specified in 'render()'.
			This function just display a message.
			
		PARAMETERS
			- gui (pygame.Surface): window	
		"""
		
		gui.draw_text_center( "Please define rendering function",
			20, (255,0,0), (255,255,255) )
	
	# =====================================================
	# Fonction principale qui réalise le rendu et la 
	# gestion des évènements
	# On utilise la couleur de fond définie dans le 
	# constructeur pour vider l'écran et on passe ensuite
	# la main à la fonction 'rendering_function'
	# =====================================================
			 
	def render( self, rendering_function = None, events_handler = None, quit_fn = None ):
	
		"""
		WHAT
			Main function which is in charge of the rendering and events
			handling. To stop the rendering you have to set the 'stop'
			variable of this class to False.
		
		PARAMETERS
		- rendering_function (function): function written by the user 
			in charge of the rendering.	This function must have one 
			parameter which is a pointer to this class
		- events_handler (function): function written by the user in 
			charge of the events. This function must have one parameter 
			which is a pointer to this class
			
		"""
		if events_handler == None:
			events_handler = self.default_events_handler
		if rendering_function == None:
			rendering_function = self.default_rendering_function

		self.delay = 10
		self.stop = False
		
		while not self.stop:
			self.window.fill( self.background_color )
			rendering_function( self )
			events_handler( self )
			pygame.display.update()
			pygame.time.delay( self.delay )
			
		if quit_fn != None:
			quit_fn()
				
		pygame.quit()
		
