#include <iostream>
#include <iomanip>
#include <cassert>
#include <cstdint>
#include <iterator>
#include <vector>
#include <algorithm>
#include <numeric>
#include <tuple>

#include <getopt.h>
#include "timer.h"

using namespace std;

enum
{
    verbose_no = 0,
    verbose_basic = 1,
    verbose_detailed = 2,
    verbose_full = 3
};

int verbose_level = verbose_basic;
int vector_size = 100;
int method = 1;

extern "C"
{
    int sum_asm(int *t, int size);
    int sum_asm_vec_sse(int *t, int size);
    int sum_asm_vec_avx(int *t, int size);
}

typedef int (*sum_fn_t)(int *t, int size);
typedef tuple<sum_fn_t, string> entry;

vector<entry> methods = {
    {nullptr, "unknown"},
    {sum_asm, "x86 assembly implementation"},
    {sum_asm_vec_sse, "x86 assembly with SSE"},
    {sum_asm_vec_avx, "x86 assembly with AVX"},
};

void usage(int argc, char *argv[])
{
    cout << endl;
    cout << argv[0] << " [options]" << endl;
    cout << endl;

    cout << "\t-method=int or -m int" << endl;
    cout << "\t\tmethod used to perform the sum" << endl;

    cout << "\t-size=int or -s int" << endl;
    cout << "\t\tsize of vector" << endl;

    cout << "\t--verbose=int or -v int" << endl;
    cout << "\t\tverbose level: 0 is equivalent to quiet" << endl;

    cout << "\t--quiet or -q" << endl;
    cout << "\t\tno output" << endl;

    cout << "\t--help or -h" << endl;
    cout << "\t\tthis message" << endl;

    cout << endl
         << endl;

    exit(EXIT_SUCCESS);
}

void parse_arguments(int argc, char *argv[])
{

    static struct option long_options[] = {

        {"verbose", required_argument, 0, 0},
        {"quiet", no_argument, 0, 0},
        {"help", no_argument, 0, 0},
        {"size", required_argument, 0, 0},
        {"method", required_argument, 0, 0},
        {0, 0, 0, 0}};

    while (true)
    {
        int option_index;
        int c = getopt_long(argc, argv, "v:qhs:m:", long_options, &option_index);
        if (c == -1)
            break;

        switch (c)
        {

        case 's':
            vector_size = atoi(optarg);
            break;

        case 'v':
            verbose_level = atoi(optarg);
            break;

        case 'q':
            verbose_level = 0;
            break;

        case 'h':
            usage(argc, argv);
            break;

        case 'm':
            method = atoi(optarg);
            break;

        default:
            throw std::runtime_error("unknown option, use -h or --help for more information");
        }
    }

    // now check parameters
    if (vector_size < 1)
    {
        throw std::runtime_error("vector size must be greater than 0");
    }

    int nbr_methods = static_cast<int>(methods.size()) - 1;

    if ((method < 1) || (method > nbr_methods))
    {
        ostringstream oss;
        oss << "method must be between 1 and " << nbr_methods;
        throw std::runtime_error(oss.str());
    }
}

int main(int argc, char *argv[])
{

    try
    {
        parse_arguments(argc, argv);
    }
    catch (exception &e)
    {
        cerr << "error: " << e.what() << endl;
        return EXIT_FAILURE;
    }

    if (verbose_level >= verbose_basic)
    {
        cout << "- create vector of size=" << vector_size << endl;
        cout << "  with values from 1 to " << vector_size << endl;
    }

    // fill vector with values 1 to size
    vector<int> v;
    for (int i = 1; i <= vector_size; ++i)
    {
        v.push_back(i);
    }

    // perform sum of values in vector
    // int sum = accumulate(v.begin(), v.end(), 0);

    if (verbose_level >= verbose_basic)
    {
        cout << "- implementation=" << std::get<1>(methods[method]) << endl;
    }

    int sum = 0;

    if (method == 1)
    {
        sum = sum_asm(v.data(), vector_size);
    }
    else if (method == 2)
    {
        sum = sum_asm_vec_sse(v.data(), vector_size);
    }
    else
    {
        sum = sum_asm_vec_avx(v.data(), vector_size);
    }

    // display result
    if (verbose_level >= verbose_detailed)
    {
        cout << "- vector contents=";
        copy(v.begin(), v.end(), ostream_iterator<int>(cout, " "));
        cout << endl;
    }

    cout << "- sum of elements=" << sum << endl;

    // put one million random values
    v.clear();
    vector_size = 1'000'000;
    v.resize(vector_size);
    srand(19702013);
    generate(v.begin(), v.end(), []()
             { return rand() % vector_size; });

    // check how much time is used to sort the vector
    timeit(cout, "sort duration", sort(v.begin(), v.end());)

        return EXIT_SUCCESS;
}
