// ==================================================================
// author: Jean-Michel Richer
// date: October, 2014
// description: basic CPU implementation
// ==================================================================

#include <unistd.h>
#include <stdlib.h>
#include <sstream>
#include <iostream>
#include <xmmintrin.h>
using namespace std;
#include <getopt.h>
#include <omp.h>

#include "gol_base.h"

#include "gl_animation.h"

int game_size = 510;
int board_size; // = game_size + 2
int max_iterations = 500;
string output_file_name = "";
int verbose = 0;
int occupation_rate = 50;
bool perform_test = false;

GLImage *image;

static struct option long_options[] = {
		{"size",  	required_argument, 0, 's'},
		{"max-iterations", 	required_argument, 0, 'm' },
		{"output", required_argument, 0, 'o'},
		{"verbose", no_argument, 0, 'v'},
		{"occupation-rate", required_argument, 0, 'r' },
		{"test", no_argument, 0, 't'},
		{0, 0, 0, 0}
};

// selector for the input (=0) or output (=1) board
int cell_selector = 0;
// boards
CellType *cpu_cells[2];


/**
 * kernel that computes next generation of the game of life
 * @param cells_in input array of (n+2)x(n+2) cells
 * @param cells_out output array of (n+2)x(n+2) cells
 * @param ptr bitmap image updated on CPU
 * @param size dimension of the game (with borders = n+2)
 */
void kernel(CellType *cells_in, CellType *cells_out, ImagePoint *img, const int size) {

	// for each line in [1,n]	
	for (int y=1; y<size-1; ++y) {
		// for each column in [1,n]
		for (int x=1; x<size-1; ++x) {

			// get values of cells surrounding the current cell (c11)
			CellType c00, c01, c02; // line y-1
			CellType c10, c11, c12; // line y
			CellType c20, c21, c22; // line y+1

			c00 = cells_in[(y-1)*size + x-1];
			c01 = cells_in[(y-1)*size + x];
			c02 = cells_in[(y-1)*size + x+1];

			c10 = cells_in[(y)*size + x-1];
			c11 = cells_in[(y)*size + x];
			c12 = cells_in[(y)*size + x+1];

			c20 = cells_in[(y+1)*size + x-1];
			c21 = cells_in[(y+1)*size + x];
			c22 = cells_in[(y+1)*size + x+1];

			// compute number of alive cells around current cell
			int nbr = c00 + c01 + c02 + c10 + c12 + c20 + c21 + c22;

			// compute new state of current cell
			c11 = (((CellType) c11) & (nbr == 2)) | (nbr == 3);

			// set new state of current cell in cells_out
			cells_out[y * size + x] = c11;

			// update image with red (ALIVE) or black(DEAD)
			int offset = y * size + x;
			img[offset].red = c11 * 255;
			img[offset].green = 0;
			img[offset].blue = 0;
			img[offset].alpha = c11 * 255;

		}
	}
}

void initialize(ImagePoint *img, int bs, CellType *cells) {
	ImagePoint *pixel = image->pixels();

	// empty first and last column
	// for the image the border will appear in green
	for (int y=0; y<bs; y++) {
		cells[y*bs] = 0;
		int offset = (y*bs);
		pixel[offset].red = 0;
		pixel[offset].green = 255;
		pixel[offset].blue = 0;
		pixel[offset].alpha = 255;
		cells[y*bs + bs-1] = 0;
		offset = (y*bs+bs-1);
		pixel[offset].red = 0;
		pixel[offset].green = 255;
		pixel[offset].blue = 0;
		pixel[offset].alpha = 255;
	}
	// empty first and last rows
	for (int x=1; x<bs-1; x++) {
		cells[x] = 0;
		int offset = (x);
		pixel[offset].red = 0;
		pixel[offset].green = 255;
		pixel[offset].blue = 0;
		pixel[offset].alpha = 255;
		cells[(bs-1)*bs + x] = 0;
		offset = ((bs-1)*bs+x);
		pixel[offset].red = 0;
		pixel[offset].green = 255;
		pixel[offset].blue = 0;
		pixel[offset].alpha = 255;
	}
	// fill center grid or GAME
	for (int y=1; y<bs-1; y++) {
		for (int x=1; x<bs-1; x++) {
			cells[y*bs+x] = ((rand() % 100) > occupation_rate) ? ALIVE : DEAD;
			int offset = (y*bs+x);
			pixel[offset].red = 255;
			pixel[offset].green = 0;
			pixel[offset].blue = 0;
			pixel[offset].alpha = 255;
		}
	}
}


/**
 * subprogram called for the animation
 */
void animate() {
	// exchange input/output cells
	int next_cell_selector = (cell_selector+1)%2;

	kernel(cpu_cells[cell_selector], cpu_cells[next_cell_selector],
			image->pixels(), board_size);

	cout << "frame=" << GLAnimation::instance().frames() <<
			",alive=" << cells_alive(cpu_cells[next_cell_selector], board_size) << endl;

	if (GLAnimation::instance().frames() == static_cast<size_t>(max_iterations)) {
		cout << "animation.frames=" << max_iterations << endl;
		cout << "cells.alive.end= " << cells_alive(cpu_cells[cell_selector], board_size) << endl;
		if (output_file_name.length() > 0) {
#ifdef LIBPNGPP
			image->save(output_file_name + ".ppm");
			image->save_png(output_file_name + ".png");
#endif
		}
		exit(EXIT_SUCCESS);
	}

	cell_selector = next_cell_selector;
}

/**
 * clean up memory allocated on the GPU
 */
void leave() {
	cout << "frames.end=" << GLAnimation::instance().frames() << endl;
	cout << "cells.alive.end= " << cells_alive(cpu_cells[cell_selector], board_size) << endl;
	free(cpu_cells[0]);
	free(cpu_cells[1]);
}

void test() {

	for (int i = 1; i <= max_iterations; ++i) {
		int next_cell_selector = (cell_selector+1)%2;
		
		kernel(cpu_cells[cell_selector], cpu_cells[next_cell_selector],
			image->pixels(), board_size);
			
		
		cell_selector = next_cell_selector;	
	}
	
	cout << "cells alive at the end=" << cells_alive(cpu_cells[cell_selector], board_size) << endl;

}

/**
 * main program
 */
int main(int argc, char *argv[]) {
	int option_index;

	// parse parameters
	while (1) {
		option_index = 0;
		int c = getopt_long (argc, argv, "s:m:o:r:vt", long_options, &option_index);
		if (c == -1) break;

		switch(c) {
		case 's':
		{
			int gs = atoi(optarg);
			if ((gs < 16) || (gs > 2048)) {
				ostringstream oss;
				oss << "vector size must be between 16 and 2048";
				throw std::runtime_error(oss.str());
			}
			game_size = gs;
		}
		break;

		case 'm':
		{
			int mi = atoi(optarg);
			if ((mi < 1) || (mi > 10000)) {
				ostringstream oss;
				oss << "maximum iterations must be between 1 and 10000";
				throw std::runtime_error(oss.str());
			}
			max_iterations = mi;
		}
		break;

		case 'o':
			output_file_name = optarg;
			break;

		case 'r':
		{
			int oc = atoi(optarg);
			if ((oc < 1) || (oc > 100)) {
				ostringstream oss;
				oss << "occupation rate must be between 1 and 100";
				throw std::runtime_error(oss.str());
			}
			occupation_rate = oc;
		}
		break;

		case 'v':
			verbose = 1;
			break;
			
		case 't':
			perform_test = true;
			break;	
		} // end switch
					
	}

	board_size = game_size + 2;

	// initialize random generator : don't change !
	srand(SRAND_SEED);

	image = new GLImage(board_size, board_size);
	GLAnimation::instance().image(image);
	
	cell_selector = 0;
	cpu_cells[0] = (CellType *) _mm_malloc(board_size * board_size * sizeof(CellType), 32);
	cpu_cells[1] = (CellType *) _mm_malloc(board_size * board_size * sizeof(CellType), 32);
	
	
	/////////////////////////////////////////////////////////////////
	// initialize the input data: cells and image
	/////////////////////////////////////////////////////////////////
	initialize(image->pixels(), board_size, cpu_cells[0]);

	cout << "cells.alive.start= " << cells_alive(cpu_cells[cell_selector], board_size) << endl;

	if (verbose) {
		cells_print(cpu_cells[cell_selector], board_size);
	}

	cout << "game_size=" << game_size << endl;
	cout << "board_size=" << board_size << endl;
	cout << "occupation_rate=" << occupation_rate << endl;

	if (perform_test) {
		test();
		exit( EXIT_SUCCESS );
	}

	GLAnimation::instance().run(argc, argv, animate,  leave);

	return 0;
}
