/*
 * command_line_argument.h
 *
 *  Created on: Apr 3, 2015
 *      Author: richer
 */

#ifndef COMMAND_LINE_ARGUMENT_PARSER_H_
#define COMMAND_LINE_ARGUMENT_PARSER_H_

#include <cstdint>
#include <map>
#include <string>
#include <vector>
#include <stdexcept>
#include <iostream>
#include <fstream>
#include <sstream>
#include <boost/lexical_cast.hpp>
using namespace std;

typedef uint32_t u32;
typedef int32_t i32;
typedef float f32;

// namespace for Command LIne Argument Parser
namespace clap {

class CommandLineArgument;

typedef void (*trigger_t)(void);

class CommandLineParser;

/**
 * default pattern to declare one class of argument
 */
class CommandLineArgument {
protected:
	// description as long label
	string long_label;
	// description as short label
	char short_label;
	// full description of argument
	string description;
	//
	bool required_value;

	bool found;
	
	trigger_t trigger; 

public:
	/**
	 * constructor with labels and description
	 * @param ll label as long format (string)
	 * @param sl label as short format (char)
	 * @param d description
	 */
	CommandLineArgument(string ll, char sl, string d, trigger_t tg = nullptr) {
		long_label = ll;
		short_label = sl;
		description = d;
		required_value = false;
		found = false;
		trigger = tg;
	}

	/**
	 * constructor with labels and description but without short argument
	 * @param ll label as long format (string)
	 * @param d description
	 * @param rv required value
	 */
	CommandLineArgument(string ll, string d, trigger_t tg = nullptr) {
		long_label = ll;
		short_label = '\0';
		description = d;
		required_value = false;
		found = false;
		trigger = tg;
	}

	/**
	 * destructor
	 */
	virtual ~CommandLineArgument() {

	}

	/**
	 * return long option as string
	 */
	string get_long_label() {
		return long_label;
	}

	/**
	 * return short option as character
	 */
	char get_short_label() {
		return short_label;
	}

	/**
	 * function used to parse argument argv[i].
	 * for example if argument is "--arg=value", only the "value"
	 * part is given as the s parameter
	 */
	virtual void parse(string& s) = 0;

	/**
	 * tells if this argument is required
	 */
	bool is_needed() {
		return required_value;
	}

	/**
	 * tells if this argument is needed
	 */
	void set_needed() {
		required_value = true;
	}

	/**
	 * sets this argument as parsed in the command line
	 */
	void set_found() {
		found = true;
	}

	/**
	 * tells if this argument was parsed in the command line
	 */
	bool was_found() {
		return found;
	}

	/**
	 * call the trigger
	 */
	void call_trigger();
	
	friend class CommandLineParser;
};

/**
 * class used to define flag argument that do not need
 * any parameter or value like -v (verbose)
 */
class FlagArgument : public CommandLineArgument {
protected:
	bool  *value;

public:
	FlagArgument(string ll, char sl, bool *v, string d, trigger_t tg = nullptr)
: CommandLineArgument(ll, sl, d, tg) {
		value = v;
	}

	void parse(string& s);
};

/**
 * class used to treat boolean argument
 */
class BooleanArgument : public CommandLineArgument {
protected:
	bool  *value;

public:
	/**
	 * constructor with pointer to boolean value to set
	 */
	BooleanArgument(string ll, char sl, bool *v, string d, trigger_t tg = nullptr)
: CommandLineArgument(ll, sl, d, tg) {
		value = v;
	}

	void parse(string& s);
};

/**
 * class used to treat integer argument
 */
class IntegerArgument : public CommandLineArgument {
protected:
	i32 *value;

public:
	IntegerArgument(string ll, char sl, i32 *v, string d, trigger_t tg = nullptr)
: CommandLineArgument(ll, sl, d, tg) {
		value = v;
	}

	void parse(string& s);
};

/**
 * class used to treat unsigned integer argument
 */
class NaturalArgument : public CommandLineArgument {
protected:
	u32 *value;

public:
	NaturalArgument(string ll, char sl, u32 *v, string d, trigger_t tg = nullptr)
: CommandLineArgument(ll, sl, d, tg) {
		value = v;
	}

	void parse(string& s);
};

/**
 * class used to treat float argument
 */
class FloatArgument : public CommandLineArgument {
protected:
	f32 *value;

public:
	FloatArgument(string ll, char sl, f32 *v, string d, trigger_t tg = nullptr)
: CommandLineArgument(ll, sl, d, tg) {
		value = v;
	}

	void parse(string& s);
};

/**
 * class used to treat float argument
 */
class RangeFloatArgument : public FloatArgument {
protected:
	f32 min_value, max_value;

public:
	RangeFloatArgument(string ll, char sl, f32 *v, f32 mini, f32 maxi, string d, trigger_t tg = nullptr)
: FloatArgument(ll, sl, v, d, tg) {
		min_value = mini;
		max_value = maxi;
	}

	void parse(string& s);
};


/**
 * class used to treat String argument
 */
class StringArgument : public CommandLineArgument {
protected:
	string *value;

public:
	StringArgument(string ll, char sl, string *v, string d, trigger_t tg = nullptr)
: CommandLineArgument(ll, sl, d, tg) {
		value = v;
	}

	void parse(string& s);
};

/**
 * class used to treat user defined or label argument.
 * A label argument will result as an integer value to set
 * in function of a list of labels
 */
class OptionsArgument : public CommandLineArgument {
protected:
	u32 *value;
	// array of allowed labels
	vector<string> *options;

public:
	OptionsArgument(string ll, char sl, u32 *v, vector<string> *opt, string d, trigger_t tg = nullptr)
: CommandLineArgument(ll, sl, d, tg) {
		value = v;
		options = opt;
	}

	void parse(string& s);

	u32  find_option(string& s);

	string get_allowed_options();

};


/**
 * main class to treat command line arguments of argv
 */
class CommandLineParser {
protected:
	// number of arguments (argc)
	int _argc;
	// arguments argv
	char **_argv;

	vector<CommandLineArgument *> arguments;

	map<string, CommandLineArgument *> map_long;
	map<char, CommandLineArgument *> map_short;

	CommandLineArgument *help_command,
		*output_command;

	string program_name;
	string program_description;
	bool _show_synopsis;
		
public:
	/**
	 * constructor with argc and argv
	 */
	CommandLineParser(string p_name, string p_desc, int argc, char *argv[]);

	~CommandLineParser();

	void clean();

	/**
	 * add new type of argument
	 * we check if the new argument to add does not have a short or long
	 * label equal to one of the existing arguments
	 * if it is a LabelArgument we check if all options are different
	 * @param arg pointer to new argument
	 */
	CommandLineArgument *add(CommandLineArgument *arg);
 
	CommandLineArgument *add_flag   (string l_label, char s_label, bool *v, string descr, trigger_t tg = nullptr);
	CommandLineArgument *add_boolean(string l_label, char s_label, bool *v, string descr, trigger_t tg = nullptr);
	CommandLineArgument *add_integer(string l_label, char s_label,  i32 *v, string descr, trigger_t tg = nullptr);
	CommandLineArgument *add_natural(string l_label, char s_label,  u32 *v, string descr, trigger_t tg = nullptr);
	CommandLineArgument *add_float  (string l_label, char s_label,  f32 *v, string descr, trigger_t tg = nullptr);
	CommandLineArgument *add_range_float(string l_label, char s_label,  f32 *v, f32 mini, f32 maxi, string descr, trigger_t tg = nullptr);
	CommandLineArgument *add_string (string l_label, char s_label, string *v, string descr, trigger_t tg = nullptr);
	CommandLineArgument *add_options(string ll, char sl, u32 *v, vector<string> *opt, string descr, trigger_t tg = nullptr);

	u32 get_nbr_arguments() {
		return arguments.size();
	}

	/**
	 * parse argv
	 * throw Exception in case of error
	 */
	void parse();
	
	void report_error(exception& e);
	
protected:
	/**
	 * print synopsis of command
	 * @param out output stream
	 * @param prog_name name of program
	 * @param descr description of program (ex for ls : list directory contents)
	 */
	void print_synopsis(ostream& out);

	bool show_synopsis() {
		return _show_synopsis;
	}
	
	void set_show_synopsis(bool v) {
		_show_synopsis = v;
	}
	
	void show_exception(string s);

	/**
	 * print string with given display length
	 */
	void print_synopsis_string(ostream& out, i32 lvl, string s, u32 length);
};

} // end of namespace

#endif /* COMMAND_LINE_ARGUMENT_PARSER_H_ */
