/*
 * command_line_argumen_parsert.cpp
 *
 *  Created on: Apr 3, 2015
 *      Author: richer
 */

#include "command_line_argument_parser.h"
using namespace clap;

static bool help_flag = false;
static ostringstream coss;
static ofstream output;
static streambuf *oldbuffer = nullptr;
static string output_file_name = "";

string line1 = "============================================================";
string line2 = "------------------------------------------------------------";

string b_underline = "\33[4m";
string e_underline = "\033[0m";
string b_bold = "\033[1m";
string e_bold = "\033[0m";

bool is_integer(string& s) {
	try {
		boost::lexical_cast<i32>(s.c_str());
	} catch (...) {
		return false;
	}
	return true;
}

bool is_natural(string& s) {
	try {
		boost::lexical_cast<u32>(s.c_str());
	} catch (...) {
		return false;
	}
	return true;
}

bool is_float(string& s) {
	try {
		boost::lexical_cast<double>(s.c_str());
	} catch (...) {
		return false;
	}
	return true;
}

bool starts_with(string& s, string& st_str) {
	return (strncmp(s.c_str(),st_str.c_str(), st_str.size())==0) ? true : false;
}

bool starts_with(string& s, char *st_str) {
	return (strncmp(s.c_str(),st_str, strlen(st_str))==0) ? true : false;
}

void trim_left(string& s, const string& pattern) {
	string::size_type pos = s.find_first_not_of(pattern);
	if (pos != string::npos) {
		s.erase(0, pos);
	} else {
		if (s.size()>0) {
			s.clear();
		}
	}
}

size_t position_of(string& s, string& st_str) {
	return static_cast<size_t>(s.find(st_str));
}

size_t position_of(string& s, char *st_str) {
	return static_cast<size_t>(s.find(st_str));
}

void CommandLineArgument::call_trigger() {
	if (trigger != nullptr) {
		(*trigger)();
	}
}

void FlagArgument::parse(string& s) {
	if (s.length() != 0) {
		throw std::logic_error("argument not allowed");
	}
	*value = true;
}

void BooleanArgument::parse(string& s) {
	if (s.length() == 0) {
		*value = true;
	} else {
		if (s == "true") {
			*value = true;
		} else if (s == "false") {
			*value = false;
		} else {
			coss << "use true or false instead of '" << s << "'";
			throw std::logic_error(coss.str());
		}
	}
}

void IntegerArgument::parse(string& s) {
	if (s.length() == 0) {
		throw std::logic_error("integer value not provided");
	} else {
		if (is_integer(s)) {
			*value = atoi(s.c_str());
		} else {
			coss << "integer value required instead of '" << s << "'";
			throw std::logic_error(coss.str());
		}
	}
}

void NaturalArgument::parse(string& s) { 
	if (s.length() == 0) {
		throw std::logic_error("natural value not provided");
	} else {
		if (is_natural(s)) {
			*value = atoi(s.c_str());
		} else {
			coss << "natural value required instead of '" << s << "'"; 
			throw std::logic_error(coss.str());
		}
	}
}

void FloatArgument::parse(string& s) {
	if (s.length() == 0) {
		throw std::logic_error("floatting point value not provided");
	} else {
		if (is_float(s)) {
			*value = atof(s.c_str());
		} else {
			coss << "floatting point value required instead of '" << s << "'";
			throw std::logic_error(coss.str());
		}
	}
}

void RangeFloatArgument::parse(string& s) {
	if (s.length() == 0) {
		throw std::logic_error("floatting point value not provided");
	} else {
		if (is_float(s)) {
			*value = atof(s.c_str());
			if ((*value < min_value) || (*value > max_value)) {
				coss << "value given " << *value << " is not in range [";
				coss << min_value << ".." << max_value << "]";
				throw std::logic_error(coss.str());
			}
		} else {
			coss << "floatting point value required instead of '" << s << "'";
			throw std::logic_error(coss.str());
		}
	}
}
void StringArgument::parse(string& s) {
	*value = s;
}


void OptionsArgument::parse(string& s) {
	if (s.length() == 0) {
		throw std::logic_error("argument not provided");
	} else {
		*value = find_option(s);
		if (*value >= options->size()) {
			coss << "option label must be chosen in: [";
			coss << get_allowed_options() << "] instead of " << s;
			throw std::logic_error(coss.str());
		}
	}
}

u32 OptionsArgument::find_option(string& s) {
	for (u32 i = 0; i < options->size(); ++i) {
		if ((*options)[i] == s) return i;
	}
	return options->size() + 1;
}

string OptionsArgument::get_allowed_options() {
	string s = (*options)[0];
	for (u32 i = 1; i < options->size(); ++i) {
		s += ", ";
		s += (*options)[i];
	}
	return s;
}

CommandLineParser::CommandLineParser(string p_name, string p_desc, int argc, char *argv[]) {
	_argc = argc;
	_argv = argv;
	program_name = p_name;
	program_description = p_desc;
	help_command = add_flag("help", 'h', &help_flag,
				"help flag, print synopsis");
	output_command = add_string("output", 'o', &output_file_name,
				"output file, used to redirect output");			
}

CommandLineParser::~CommandLineParser() {
	for (auto arg : arguments) {
		delete arg;
	}
	map_long.clear();
	map_short.clear();
	if (oldbuffer != nullptr) cout.rdbuf(oldbuffer);
}


void CommandLineParser::report_error(exception& e) {
	if (show_synopsis()) {
		print_synopsis(cout);
	}
		
	show_exception(e.what());
	exit(EXIT_FAILURE);
}

	
void CommandLineParser::clean() {
	for (auto a : arguments) {
		delete a;
	}
}

CommandLineArgument *CommandLineParser::add(CommandLineArgument *arg) {
	for (auto a : arguments) {
		if (a->get_long_label() == arg->get_long_label()) {
			coss << "label " << a->get_long_label() << " already exists,";
			coss << "choose another label";
			throw std::logic_error(coss.str()); 
		}
		if (arg->get_short_label() != '\0') {
			if (a->get_short_label() == arg->get_short_label()) {
				coss << "label " << a->get_short_label() << " alreday exists,";
				coss << "choose another label";
				throw std::logic_error(coss.str()); 
			}
		}
	}
	arguments.push_back(arg);
	map_long[arg->get_long_label()] = arg;
	if (arg->get_short_label() != '\0') {
		map_short[arg->get_short_label()] = arg;
	}
	return arg;
}

CommandLineArgument *CommandLineParser::add_flag(string l_label, char s_label, bool *v, string descr, trigger_t tg) {
	CommandLineArgument *arg = new FlagArgument(l_label, s_label, v, descr, tg);
	return add(arg);
}

CommandLineArgument *CommandLineParser::add_boolean(string l_label, char s_label, bool *v, string descr, trigger_t tg) {
	CommandLineArgument *arg = new FlagArgument(l_label, s_label, v, descr, tg);
	return add(arg);
}

CommandLineArgument *CommandLineParser::add_integer(string l_label, char s_label, i32 *v, string descr, trigger_t tg) {
	CommandLineArgument *arg = new IntegerArgument(l_label, s_label, v, descr, tg);
	return add(arg);
}

CommandLineArgument *CommandLineParser::add_natural(string l_label, char s_label, u32 *v, string descr, trigger_t tg) {
	CommandLineArgument *arg = new NaturalArgument(l_label, s_label, v, descr, tg);
	return add(arg);
}

CommandLineArgument *CommandLineParser::add_float(string l_label, char s_label, f32 *v, string descr, trigger_t tg) {
	CommandLineArgument *arg = new FloatArgument(l_label, s_label, v, descr, tg);
	return add(arg);
}

CommandLineArgument *CommandLineParser::add_range_float(string l_label, char s_label,  f32 *v,
		f32 mini, f32 maxi, string descr, trigger_t tg) {
	CommandLineArgument *arg = new RangeFloatArgument(l_label, s_label, v, mini, maxi, descr, tg);
	return add(arg);
}

CommandLineArgument *CommandLineParser::add_string(string l_label, char s_label, string *v, string descr, trigger_t tg) {
	CommandLineArgument *arg = new StringArgument(l_label, s_label, v, descr, tg);
	return add(arg);
}

CommandLineArgument *CommandLineParser::add_options(string l_label, char s_label, u32 *v, vector<string> *opt, string descr, trigger_t tg) {
	CommandLineArgument *arg = new OptionsArgument(l_label, s_label, v, opt, descr, tg );
	return add(arg);
}
	
void CommandLineParser::parse() {
	int i = 1;

	_show_synopsis = true;
	string long_option_intro = "--";
	string short_option_intro = "-";
	 
	while (i < _argc) {
		string arg = _argv[i];
		string value = "";

		if (starts_with(arg, long_option_intro)) {
			string label = "";

			trim_left(arg, short_option_intro);

			string s_equal = "=";
			size_t pos = position_of(arg, s_equal);

			if (pos != string::npos) {
				label = arg.substr(0, pos);
				if (pos + 1 < arg.length()) {
					value = arg.substr(pos+1);
				}
			} else {
				label = arg;
			}

			CommandLineArgument *argument = nullptr;
			auto it_long = map_long.find(label);
			if (it_long != map_long.end()) {
				argument = (*it_long).second;
			} else {
				coss << "label \"" + label + "\" does not exist";
				throw std::logic_error(coss.str());
			}
			try {
				argument->parse(value);
			} catch(exception& e) {
				coss.str("");
				coss << e.what();
				coss << ", for argument " << _argv[i];
				throw std::logic_error(coss.str());
			}
			argument->set_found();
			

		} else if (starts_with(arg, short_option_intro)) {
			char label;

			trim_left(arg, short_option_intro);
			if (arg.length() != 1) {
				coss << "command line argument \"" << _argv[i]; 
				coss << "\" is in short format and needs only one character";
				throw std::logic_error(coss.str());
			}
			label = arg[0];
			CommandLineArgument *argument = nullptr;
			auto it_short = map_short.find(label);
			if (it_short != map_short.end()) {
				argument = (*it_short).second;
			} else {
				coss << "label \"" << label << "\" does not exist";
				throw std::logic_error(coss.str());
			}
			if (typeid(*argument) != typeid(FlagArgument)) {
				++i;
				if (i == _argc) {
					coss << "no value for argument -" << label;
					throw std::logic_error(coss.str());
				}
				value = _argv[i];
				argument->parse(value);
				argument->set_found();
			} else {
				string empty = "";
				argument->parse(empty);
				argument->set_found();
			}

		} else {
			coss << "argument does not follow short or long format: \"" << _argv[i];
			coss << "\", use - or -- before argument label";
			throw std::logic_error(coss.str());
		}
		++i;
	}

	if (help_command->was_found()) {
		print_synopsis(cout);
		exit(EXIT_SUCCESS);
	}
	help_flag = false;
	
	// check that needed command line options were parsed
	for (auto arg : arguments) {
		if (arg->is_needed()) {
			if (!arg->was_found()) {
				ostringstream oss;
				oss << "command line option \"" << arg->get_long_label() << "\" needs to be defined";
				throw std::logic_error(oss.str());
			}
		}
	}
			
	// launch triggers
	_show_synopsis = false;
	for (auto arg : arguments) {
		if (arg->was_found()) {
			arg->call_trigger();
		}
	}
	
	// redirects output if needed
	if (output_file_name.length() > 0) {
		oldbuffer = cout.rdbuf(output.rdbuf());
		output.open(output_file_name.c_str());
		if (!output) {
			ostringstream oss;
			oss << "could not open file \"" << output_file_name << " \"for writing";
			throw std::logic_error(oss.str());
		}
	}
	
}

void CommandLineParser::print_synopsis_string(ostream& out, i32 lvl, string s, u32 length) {
	u32 l = 0;
	istringstream iss(s);
	string word;

	for (i32 i = 0; i < lvl; ++i) out << "\t";
	while (iss >> word) {
		if ((l + word.size()) > length) {
			out << endl;
			for (int i = 0; i < lvl; ++i)	out	<< "\t";
			l = 0;
		}
		out << word << " ";
		l += word.size() + 1;
	}
	out << endl;
}

void add_to_synopsis(ostream& out, CommandLineArgument *arg, string type) {
	out << b_bold << "\t--" << arg->get_long_label() << e_bold;
	if (type.length() != 0) {
		out << "=" << b_underline << type << e_underline;
	}
	if (arg->get_short_label() != '\0') {
		out << " or " << b_bold << "-" << arg->get_short_label() << e_bold;
		if (type.length() != 0) {
			out << " " << b_underline << type << e_underline;
		}
	}
}

void CommandLineParser::print_synopsis(ostream& out) {
	out << b_bold << "NAME" << e_bold << endl;
	ostringstream oss;
	oss << program_name << " - " << program_description;
	print_synopsis_string(out, 1, oss.str(), 60);
	out << endl << b_bold << "SYNOPSIS" << e_bold << endl ;
	oss.str("");
	oss << program_name << " [OPTIONS] ... " ;
	print_synopsis_string(out, 1, oss.str(), 60);
	out << endl << b_bold << "DESCRIPTION" << e_bold << endl << endl;

	sort(arguments.begin(), arguments.end(), [](CommandLineArgument *a1, CommandLineArgument *a2) {
		return a1->get_long_label() < a2->get_long_label();
	});
	for (auto arg : arguments) {

		string value_type = "";
		if (typeid(*arg) == typeid(FlagArgument)) {
			add_to_synopsis(out, arg, "");
		} else if (typeid(*arg) == typeid(BooleanArgument)) {
			add_to_synopsis(out, arg, "BOOLEAN");
		} else if (typeid(*arg) == typeid(IntegerArgument)) {
			add_to_synopsis(out, arg, "INTEGER");
		} else if (typeid(*arg) == typeid(NaturalArgument)) {
			add_to_synopsis(out, arg, "NATURAL");
		} else if (typeid(*arg) == typeid(StringArgument)) {
			add_to_synopsis(out, arg, "STRING");
		} else if (typeid(*arg) == typeid(FloatArgument)) {
			add_to_synopsis(out, arg, "FLOAT");
		} else if (typeid(*arg) == typeid(OptionsArgument)) {
			add_to_synopsis(out, arg, "VALUE");
			out << "\n\t  where value=" << dynamic_cast<OptionsArgument *>(arg)->get_allowed_options();
		}
		out << endl;
		print_synopsis_string(out, 2, arg->description, 60);
		out << endl;
	}
}

void CommandLineParser::show_exception(string s) {
	cerr << line1 << endl;
	cerr << "Exception raised in program " << program_name << endl;
	cerr << line1 << endl;
	print_synopsis_string(cerr, 0, s, 60);
	cerr << line1 << endl;
	exit(EXIT_FAILURE);
}
