/*
 * strings.h
 *
 *  Created on: Apr 2, 2015
 *      Author: richer
 */

#ifndef STRINGS_H_
#define STRINGS_H_

#include "exception.h"
#include <functional>
#include <cctype>
#include <sstream>
#include <algorithm>
#include <vector>
#include <string>
#include <cstring>
#include <boost/lexical_cast.hpp>

using namespace std;

namespace base {

#define STRING_BLANKS " \f\n\r\t\v"
#define STRING_BLANK  " "

class Strings {
public:
	/**
	 * remove all characters that correspond to any character of pattern
	 * at the end of the string
	 */
	static void trim_right(string& s, const string& pattern=STRING_BLANKS);

	/**
	 * remove all characters that correspond to any character of pattern
	 * at beginning of the string
	 */
	static void trim_left(string& s, const string& pattern=STRING_BLANKS);

	/**
	 * remove all characters that correspond to any character of pattern
	 * at beginning and end of the string
	 */
	static void trim(string& s, const string& pattern=STRING_BLANKS);

	/**
	 * erase all characters that correspond to any character of pattern
	 */
	static void erase(string& s, const string& pattern=STRING_BLANKS);

	/**
	 * remove all spaces characters
	 */
	static void remove_spaces(string& s);

	/**
	 * replace all occurrences of source character by replacement character
	 */
	static void replace(string& s, char src, char repl);

	/**
	 * convert string to lowercase
	 */
	static void to_lower(string& s);

	/**
	 * convert string to uppercase
	 */
	static void to_upper(string& s);

	/**
	 * convert string represented in C format to lowercase
	 */
	static void to_lower(char *s);

	/**
	 * convert string represented in C format to uppercase
	 */
	static void to_upper(char *s);

	/**
	 * split string in substrings when separated by any character of delimiters
	 * the substrings are put in a vector
	 * @param s string to explode
	 * @param v output substrings
	 * @param delim characters used as delimiters
	 */
	static void explode(string& s, vector<string>& v, const string& delim=STRING_BLANKS);

	/**
	 * join strings of a vector into a single string where delimiters separate
	 * each substring
	 * @param s string used as a result
	 * @param v vector of strings to implode
	 *
	 */
	static void implode(string& s, vector<string>& v, const string& delim=STRING_BLANK);
	static void implode(string& s, vector<char>& v, const string& delim=STRING_BLANK);

	/**
	 * return position of st_str in s or string::npos if not found
	 * @param s string that will be searched
	 * @param st_str substring searched in s
	 * @return position of st_str in s, pr string::npos
	 */
	static size_t position_of(string& s, string& st_str);
	/**
	 * return position of st_str in s or string::npos if not found
	 * @param s string that will be searched
	 * @param st_str substring searched in s
	 * @return position of st_str in s, pr string::npos
	 */
	static size_t position_of(string& s, char *st_str);

	/**
	 *
	 */
	static void cut(string& s, size_t pos);

	/**
	 * check if string s starts with substring st_str
	 * @param s main string
	 * @param st_str substring that might start s
	 * @return true if st_str starts s, false otherwise
	 */
	static bool starts_with(string& s, string& st_str);
	static bool starts_with(string& s, char *st_str);
	static bool starts_with(string& s, const char *st_str);
	static bool ends_with(string& s, string& st_str);
	static bool ends_with(string& s, char *st_str);
	static bool ends_with(string& s, const char *st_str);

	static bool is_integer(string& s);
	static bool is_natural(string& s);
	static bool is_float(string& s);
	static bool is_numeric(string& s);
	static int  to_int(string& s);
	static float to_float(string& s);
	static double to_double(string& s);
};

}



#endif /* STRINGS_H_ */
