/*
 * statistics.h
 *
 *  Created on: Apr 21, 2015
 *      Author: richer
 */

#ifndef STATISTICS_H_
#define STATISTICS_H_

#include <cmath>
#include <vector>
#include <algorithm>
#include <iterator>
#include <numeric>
#include <iostream>
using namespace std;
#include "base/types.h"

namespace maths {

enum {
	PRINT_SUM = 1, // sum
	PRINT_AVG = 2, // average
	PRINT_VAR = 4, // variance
	PRINT_SDV = 8, // standard deviation
	PRINT_MIN = 16, // minimum value
	PRINT_MAX = 32, // maximum value
	PRINT_MED = 64, // median
	PRINT_CENT = 128, // centiles
	PRINT_FREQ_MIN = 256,
	PRINT_FREQ_MAX = 512,
	PRINT_ALL = 1023
};

/**
 * @class Statistics
 * @brief template class used to compute statistics on a series of data
 * @details
 * <ul>
 *	<li>DataType is the type of input data (ex: int)</li>
 * 	<li>RealType is the type of output data (float or double)</li>
 * </ul>
 */
template<class DataType, class RealType>
class Statistics {
public:
	
	vector<DataType> values; // series of values
	DataType min, max; // minimum and maximum
	base::u32 fmin, fmax; // frequencies of min and max values
	DataType centiles[10]; // centiles
	DataType sum; // sum
	RealType avg; // average
	RealType var; // variance
	RealType sdv; // standard deviation
	RealType med; // mediane
	base::u32 print_flags;


	/**
	 * constructor with inital data
	 * Description: data are copied into
	 * @param v vector of data
	 */
	Statistics(vector<DataType>& v, base::u32 _print_flags = PRINT_ALL) {
		copy(v.begin(), v.end(), back_inserter(values));
		sort(values.begin(), values.end());
		print_flags = _print_flags;
		fmin = fmax = 0;
	}

	/**
	 * Inner class used as functor to compute standard deviation
	 */
	class Insider {
	public:
		RealType average, summer;
		int nbr;

		Insider(RealType a, int n) : average(a), summer(0.0), nbr(n) {
		}
		void operator()(DataType v) {
			summer += (v - average) * (v - average);
		}
		RealType get_variance() {
			return summer / nbr;
		}
	};

	/**
	 * main method used to compute average, ...
	 */
	void compute() {
		sum = accumulate(values.begin(), values.end(), 0);
		avg = static_cast<RealType>(sum) / values.size();
		Insider insider(avg, values.size());
		insider = for_each(values.begin(), values.end(), insider);
		var = insider.get_variance();
		sdv = sqrt(var / values.size());

		min = values[0];
		max = values[values.size()-1];

		fmin = count(values.begin(), values.end(), min);
		fmax = count(values.begin(), values.end(), max);

		if ((values.size() % 2) == 0) {
			med = static_cast<RealType>(values[values.size() / 2 - 1] + values[values.size() / 2]) / 2.0;
		} else {
			med = static_cast<RealType>(values[values.size() / 2]);
		}

		for (int i=1; i<10; ++i) {
			int index = (10 * i * values.size())/100;
			centiles[i] = values[index];
		}
	}

	RealType get_average() {
		return avg;
	}

	RealType get_variance() {
		return var;
	}

	RealType get_standard_deviation() {
		return sdv;
	}

	ostream& print(ostream& out) {
		out << std::fixed;
		if ((print_flags & PRINT_SUM) != 0) out << "sum = " << sum << endl;
		if ((print_flags & PRINT_AVG) != 0) out << "avg = " << avg << endl;
		if ((print_flags & PRINT_VAR) != 0) out << "var = " << var << endl;
		if ((print_flags & PRINT_SDV) != 0) out << "sdv = " << sdv << endl;
		if ((print_flags & PRINT_MIN) != 0) out << "min = " << min << endl;
		if ((print_flags & PRINT_FREQ_MIN) != 0) out << "fmin = " << fmin << endl;
		if ((print_flags & PRINT_MAX) != 0) out << "max = " << max << endl;
		if ((print_flags & PRINT_FREQ_MAX) != 0) out << "fmax = " << fmax << endl;
		if ((print_flags & PRINT_MED) != 0) out << "med = " << med << endl;
		if ((print_flags & PRINT_CENT) != 0) {
			for (int i=1;i<10;++i) {
				out << "centiles[" << i*10 << "] = " << centiles[i] << endl;
			}
		}
		return out;
	}

	friend ostream& operator<<(ostream& out, Statistics<DataType,RealType>& s) {
		return s.print(out);
	}
};

} // end of namespace

#endif /* STATISTICS_H_ */
