#include "test_strings.h"
#include <algorithm>
#include <iterator>
#include <numeric>
using namespace std;
#include <getopt.h>
#include <cstring>
#include <signal.h>
#include <err.h>
#include <cstdlib>
#include <cstdio>
 
void signal_handler(int sig) {
	cerr << endl << "caught signal " << sig << endl;
	cerr.flush();  
	return ;
}

void set_signal_handler() {
 	signal( SIGABRT, signal_handler);
 	signal( SIGHUP, signal_handler);
 	signal( SIGILL, signal_handler);
 }
  
// ----------------------------------------------
// description of possible arguments
// that can be used in
// - short format : -m 1
// - or long format : --method=1
// ----------------------------------------------
static struct option long_options[] = {
  {"nbr-tests",  		no_argument, 0, 'n'},
  {"output-format",  	required_argument, 0, 'f'},
  {0,0,0,0}
};

typedef struct {
	string output_file;
	int  output_format;
	bool flag_get_nbr_tests;
} Parameters;

Parameters params;

const char *params_format_allowed[] = { "text", "xml", NULL };

int get_allowed_value(const char *s, const char *tab[]) {
	for (int i=0; tab[i] != NULL; ++i) {
		if (strcmp(s,tab[i])==0) return i;
	}
	return -1;
}

void usage(char *prog_name) {
	cout << prog_name << endl;
	cout << "--output-format=text|xml" << endl;
	cout << "--nbr-tests" << endl;
	
	exit(EXIT_FAILURE);
}

// ----------------------------------------------

/**
 * setUp: function called before each test
 */
void TestFixture::setUp() {
	
}

/**
 * setUp: function called after each test
 */
void TestFixture::tearDown() {
	
}

/**
 *
 */
void TestFixture::test_is_integer() {
	vector<string> patterns({ "0", "1", "-5452", "1.1", "3;3", "a0" });
	vector<bool> expected_results({ true, true, true, false, false, false}); 
	
	int i = 0;
	for (auto s : patterns) {
		//cerr << s << "? " << Strings::is_integer(s) << " " << expected_results[i] << endl;
		CPPUNIT_ASSERT(Strings::is_integer(s) == expected_results[i]);	
		++i;
	} 
}

/**
 * 
 */
void TestFixture::test_is_float() {
	vector<string> patterns({ "0", "1", "-5452", "0.1", "3;3", "a0" });
	vector<bool> expected_results({ true, true, true, false, false, false});
	
	int i = 0;
	for (auto s : patterns) {
		cerr << s << "? " << Strings::is_float(s) << " " << expected_results[i] << endl;
		//CPPUNIT_ASSERT(Strings::is_float(s) == expected_results[i]);	
		++i;
	}	
}

/**
 * declare suite of tests
 *
 */
CppUnit::TestSuite *make_suite() {
	CppUnit::TestSuite *suite = new CppUnit::TestSuite(CLASS_NAME_STRING);
	cout << "==============================================" << endl;
	cout << "TEST " << suite->getName() << " (" << __FILE__ << ")" << endl;
	cout << "==============================================" << endl;

	ADD_TEST(is_integer);
	ADD_TEST(is_float);

	return suite;
}

/**
 * main function
 */
int main(int argc, char *argv[]) {
	CppUnit::TextUi::TestRunner runner;
	CppUnit::XmlOutputter *xml_outputter = NULL;

	params.output_file = "";
	params.output_format = 0; // text
	params.flag_get_nbr_tests = false;
	
	int option_index;
	while (true) {
    	option_index = 0;
        int c = getopt_long(argc, argv, "f:n", long_options, &option_index);
        if (c == -1) break;
 
		switch(c) {
		  case 'n':
		    params.flag_get_nbr_tests = true;
		    break;
		  case 'f':
		    params.output_format = get_allowed_value(optarg, params_format_allowed);
		    break;
		}
	}
  	
  	if (params.output_format == -1) {
  		usage(argv[0]);
  	}
  	
  	
    set_signal_handler();

	// create suite
	CppUnit::TestSuite *suite = make_suite(); 
	runner.addTest(suite);
	
	if (params.output_format == 1) {
		cout << "xml file=" << OUTPUT_XML_FILE << endl;
 	}
 	
	if (params.flag_get_nbr_tests == true) {
		const std::vector<CppUnit::Test *>& tests = suite->getTests();
		cout << "nbr_tests=" << tests.size() << endl;
		exit(EXIT_SUCCESS);
	}
	 
	// set output format as text
	runner.setOutputter(new CppUnit::CompilerOutputter(&runner.result(), cout));
	
	// run all tests
	runner.run();
	
	if (params.output_format == 1) {
		ofstream xml_out(OUTPUT_XML_FILE);
		xml_outputter = new CppUnit::XmlOutputter(&runner.result(), xml_out);
		xml_outputter->write();
		xml_out.close();
	}
	
	return 0;
}

