/*
 * command_line_argument.h
 *
 *  Created on: Apr 3, 2015
 *      Author: richer
 */

#ifndef COMMAND_LINE_ARGUMENT_H_
#define COMMAND_LINE_ARGUMENT_H_

#include <map>
#include <boost/lexical_cast.hpp>
#include "exception.h"
#include "terminal.h"
#include "strings.h"

namespace base {

class CommandLine;

/**
 * default pattern to declare one class of argument
 */
class CommandLineArgument {
protected:
	// description as long label
	string long_label;
	// description as short label
	char short_label;
	// full description of argument
	string description;
	//
	bool required_value;

public:
	/**
	 * constructor with labels and description
	 * @param ll label as long format (string)
	 * @param sl label as short format (char)
	 * @param d description
	 */
	CommandLineArgument(string ll, char sl, string d, bool rv) {
		long_label = ll;
		short_label = sl;
		description = d;
		required_value = rv;
	}

	/**
	 * constructor with labels and description but without short argument
	 * @param ll label as long format (string)
	 * @param sl label as short format (char)
	 * @param d description
	 */
	CommandLineArgument(string ll, string d, bool rv) {
		long_label = ll;
		short_label = '\0';
		description = d;
		required_value = rv;
	}

	virtual ~CommandLineArgument() {

	}

	string get_long_label() {
		return long_label;
	}

	char get_short_label() {
		return short_label;
	}

	/**
	 * function used to parse argument argv[i].
	 * for example if argument is "--arg=value", only the "value"
	 * part is given as the s parameter
	 */
	virtual void parse(string& s) = 0;

	/**
	 * tells if this argument is required
	 */
	bool need_value() {
		return required_value;
	}

	void set_needed() {
		required_value = true;
	}

	friend class CommandLine;
};

/**
 * class used to define flag argument that do not need
 * any parameter or value like -v (verbose)
 */
class FlagArgument : public CommandLineArgument {
protected:
	bool  *value;

public:
	FlagArgument(string ll, char sl, string d, bool *v)
: CommandLineArgument(ll, sl, d, false) {
		value = v;
	}

	void parse(string& s);
};

/**
 * class used to treat boolean argument
 */
class BooleanArgument : public CommandLineArgument {
protected:
	bool  *value;

public:
	/**
	 * constructor with pointer to boolean value to set
	 */
	BooleanArgument(string ll, char sl, string d, bool *v)
: CommandLineArgument(ll, sl, d, true) {
		value = v;
	}

	void parse(string& s);
};

/**
 * class used to treat integer argument
 */
class IntegerArgument : public CommandLineArgument {
protected:
	i32 *value;

public:
	IntegerArgument(string ll, char sl, string d, int *v)
: CommandLineArgument(ll, sl, d, true) {
		value = v;
	}

	void parse(string& s);
};

/**
 * class used to treat unsigned integer argument
 */
class NaturalArgument : public CommandLineArgument {
protected:
	u32 *value;

public:
	NaturalArgument(string ll, char sl, string d, u32 *v)
: CommandLineArgument(ll, sl, d, true) {
		value = v;
	}

	void parse(string& s);
};

/**
 * class used to treat float argument
 */
class FloatArgument : public CommandLineArgument {
protected:
	f32 *value;

public:
	FloatArgument(string ll, char sl, string d, f32 *v)
: CommandLineArgument(ll, sl, d, true) {
		value = v;
	}

	void parse(string& s);
};

/**
 * class used to treat float argument
 */
class RangeFloatArgument : public FloatArgument {
protected:
	f32 min_value, max_value;

public:
	RangeFloatArgument(string ll, char sl, string d, f32 *v, f32 mini, f32 maxi)
: FloatArgument(ll, sl, d, v) {
		min_value = mini;
		max_value = maxi;
	}

	void parse(string& s);
};


/**
 * class used to treat String argument
 */
class StringArgument : public CommandLineArgument {
protected:
	string *value;

public:
	StringArgument(string ll, char sl, string d, string *v)
: CommandLineArgument(ll, sl, d, true) {
		value = v;
	}

	void parse(string& s);
};

/**
 * class used to treat user defined or label argument.
 * A label argument will result as an integer value to set
 * in function of a list of labels
 */
class OptionsArgument : public CommandLineArgument {
protected:
	u32 *value;
	// array of allowed labels
	vector<string> *options;

public:
	OptionsArgument(string ll, char sl, string d, u32 *v, vector<string> *opt)
: CommandLineArgument(ll, sl, d, true) {
		value = v;
		options = opt;
	}

	void parse(string& s);

	u32  find_option(string& s);

	string get_allowed_options();

};


/**
 * main class to treat command line arguments of argv
 */
class CommandLine {
protected:
	// number of arguments (argc)
	int _argc;
	// arguments argv
	char **_argv;

	vector<CommandLineArgument *> arguments;

	map<string, CommandLineArgument *> map_long;
	map<char, CommandLineArgument *> map_short;

	bool _show_synopsis;
	
public:
	/**
	 * constructor with argc and argv
	 */
	CommandLine(int argc, char *argv[]) {
		_argc = argc;
		_argv = argv;
		_show_synopsis = true;
	}

	~CommandLine() {
		for (auto arg : arguments) {
			delete arg;
		}
		map_long.clear();
		map_short.clear();
	}

	string get_program_name() {
		return _argv[0];
	}
	
	/**
	 * add new type of argument
	 * we check if the new argument to add does not have a short or long
	 * label equal to one of the existing arguments
	 * if it is a LabelArgument we check if all options are different
	 * @param arg pointer to new argument
	 */
	void add(CommandLineArgument *arg);

	void set_show_synopsis(bool v) {
		_show_synopsis = v;
	}
	
	bool get_show_synopsis() {
		return _show_synopsis;
	}
	
	u32 get_nbr_arguments() {
		return arguments.size();
	}

	/**
	 * parse argv
	 * throw Exception in case of error
	 */
	void parse();

	/**
	 * print synopsis of command
	 * @param out output stream
	 * @param prog_name name of program
	 * @param descr description of program (ex for ls : list directory contents)
	 */
	void print_synopsys(ostream& out, string descr);

protected:

	/**
	 * print string with given display length
	 */
	void print_synopsis_string(ostream& out, i32 lvl, string s, u32 length);
};

} // end of namespace

#endif 

/* COMMAND_LINE_ARGUMENT_H_ */
