/*
 * command_line_argument.cpp
 *
 *  Created on: Apr 3, 2015
 *      Author: richer
 */

#include "command_line.h"

using namespace base;

void FlagArgument::parse(string& s) {
	if (s.length() != 0) {
		throw_t(EXC_ARGUMENT_NOT_NEEDED);
	}
	*value = true;
}

void BooleanArgument::parse(string& s) {
	if (s.length() == 0) {
		*value = true;
	} else {
		if (s == "true") {
			*value = true;
		} else if (s == "false") {
			*value = false;
		} else {
			throw_c(EXC_BAD_VALUE, "use true or false");
		}
	}
}

void IntegerArgument::parse(string& s) {
	if ((s.length() != 0) && (Strings::is_integer(s))) {
		*value = atoi(s.c_str());
	} else {
		throw_c(EXC_ARGUMENT_REQUIRED, "integer required");
	} 
}

void NaturalArgument::parse(string& s) {
	if ((s.length() != 0) && (Strings::is_natural(s))) {
		*value = static_cast<u32>(atol(s.c_str()));
	} else {
		throw_c(EXC_ARGUMENT_REQUIRED, "natural required");
	}
}

void FloatArgument::parse(string& s) {
	if ((s.length() != 0) && (Strings::is_float(s))) {
		*value = atof(s.c_str());
	} else {
		throw_c(EXC_ARGUMENT_REQUIRED, "floating point value required");
	}
}

void RangeFloatArgument::parse(string& s) {
	if (s.length() == 0) {
		throw_c(EXC_ARGUMENT_REQUIRED, "float required");
	} else {
		if (Strings::is_float(s)) {
			*value = atof(s.c_str());
			if ((*value < min_value) || (*value > max_value)) {
				throw_c(EXC_OUT_OF_RANGE, "value given is " << *value << " is not in range ["
					<< min_value << ".." << max_value << "]");
			}
		} else {
			throw_c(EXC_BAD_TYPE, "float required");
		}
	}
}
void StringArgument::parse(string& s) {
	*value = s;
}


void OptionsArgument::parse(string& s) {
	if (s.length() == 0) {
		throw_c(EXC_ARGUMENT_REQUIRED, "float required");
	} else {
		*value = find_option(s);
		if (*value >= options->size()) {
			throw_c(EXC_OUT_OF_RANGE,  "use one of: " << get_allowed_options());
		}
	}
}

u32 OptionsArgument::find_option(string& s) {
	for (u32 i = 0; i < options->size(); ++i) {
		if ((*options)[i] == s) return i;
	}
	return options->size() + 1;
}

string OptionsArgument::get_allowed_options() {
	string s = (*options)[0];
	for (u32 i = 1; i < options->size(); ++i) {
		s += ", ";
		s += (*options)[i];
	}
	return s;
}


void CommandLine::add(CommandLineArgument *arg) {
	for (auto a : arguments) {
		if (a->get_long_label() == arg->get_long_label()) {
			throw_c(EXC_ALREADY_EXIST, "label " << a->get_long_label() << "\nchoose another label");
		}
		if (arg->get_short_label() != '\0') {
			if (a->get_short_label() == arg->get_short_label()) {
				throw_c(EXC_ALREADY_EXIST, "label " << a->get_short_label() << "\nchoose another label");
			}
		}
	}
	arguments.push_back(arg);
	map_long[arg->get_long_label()] = arg;
	if (arg->get_short_label() != '\0') {
		map_short[arg->get_short_label()] = arg;
	}
}

void CommandLine::parse() {
	int i = 1;

	while (i < _argc) {
		string arg = _argv[i];
		string value = "";

		if (Strings::starts_with(arg, "--")) {
			string label = "";

			Strings::trim_left(arg, "-");

			string s_equal = "=";
			size_t pos = Strings::position_of(arg, s_equal);

			if (pos != string::npos) {
				label = arg.substr(0, pos);
				if (pos + 1 < arg.length()) {
					value = arg.substr(pos+1);
				}
			} else {
				label = arg;
			}

			CommandLineArgument *argument = nullptr;
			auto it_long = map_long.find(label);
			if (it_long != map_long.end()) {
				argument = (*it_long).second;
			} else {
				throw_c(EXC_DOES_NOT_EXIST, "label " + label);
			}
			argument->parse(value);

		} else if (Strings::starts_with(arg, "-")) {
			char label;

			Strings::trim_left(arg, "-");
			if (arg.length() != 1) {
				throw_c(EXC_BAD_VALUE, "command line argument is in short format and needs only one character");
			}
			label = arg[0];
			CommandLineArgument *argument = nullptr;
			auto it_short = map_short.find(label);
			if (it_short != map_short.end()) {
				argument = (*it_short).second;
			} else {
				throw_c(EXC_DOES_NOT_EXIST, "label " + label);
			}
			if (typeid(*argument) != typeid(FlagArgument)) {
				++i;
				if (i == _argc) {
					throw_c(EXC_ARGUMENT_REQUIRED, "no value for argument -" << label);
				}
				value = _argv[i];
				argument->parse(value);
			}
		} else {
			throw_c(EXC_GENERAL, "argument does not follow short or long format: " << arg 
				<< "\nuse - or -- before argument label");
		}
		++i;
	}

}

void CommandLine::print_synopsis_string(ostream& out, i32 lvl, string s, u32 length) {
	u32 l = 0;
	istringstream iss(s);
	string word;

	for (i32 i = 0; i < lvl; ++i) out << "\t";
	while (iss >> word) {
		if ((l + word.size()) > length) {
			out << endl;
			for (int i = 0; i < lvl; ++i)	out	<< "\t";
			l = 0;
		}
		out << word << " ";
		l += word.size() + 1;
	}
	out << endl;
}

void add_to_synopsis(ostream& out, CommandLineArgument *arg, string type) {
	out << Terminal::b_bold << "\t--" << arg->get_long_label() << Terminal::e_bold;
	if (type.length() != 0) {
		out << "=" << Terminal::b_underline << type << Terminal::e_underline;
	}
	if (arg->get_short_label() != '\0') {
		out << " or " << Terminal::b_bold << "-" << arg->get_short_label() << Terminal::e_bold;
		if (type.length() != 0) {
			out << " " << Terminal::b_underline << type << Terminal::e_underline;
		}
	}
}

void CommandLine::print_synopsys(ostream& out, string descr) {
	if (!_show_synopsis) return ;
	
	out << Terminal::b_bold << "NAME" << Terminal::e_bold << endl;
	ostringstream oss;
	oss << get_program_name() << " - " << descr;
	print_synopsis_string(out, 1, oss.str(), 60);
	out << endl << Terminal::b_bold << "SYNOPSIS" << Terminal::e_bold << endl ;
	oss.str("");
	oss << get_program_name() << " [OPTIONS] ... " ;
	print_synopsis_string(out, 1, oss.str(), 60);
	out << endl << Terminal::b_bold << "DESCRIPTION" << Terminal::e_bold << endl << endl;

	sort(arguments.begin(), arguments.end(), [](CommandLineArgument *a1, CommandLineArgument *a2) {
		return a1->get_long_label() < a2->get_long_label();
	});
	for (auto arg : arguments) {

		string value_type = "";
		if (typeid(*arg) == typeid(FlagArgument)) {
			add_to_synopsis(out, arg, "");
		} else if (typeid(*arg) == typeid(BooleanArgument)) {
			add_to_synopsis(out, arg, "BOOLEAN");
		} else if (typeid(*arg) == typeid(IntegerArgument)) {
			add_to_synopsis(out, arg, "INTEGER");
		} else if (typeid(*arg) == typeid(NaturalArgument)) {
			add_to_synopsis(out, arg, "NATURAL");
		} else if (typeid(*arg) == typeid(StringArgument)) {
			add_to_synopsis(out, arg, "STRING");
		} else if (typeid(*arg) == typeid(FloatArgument)) {
			add_to_synopsis(out, arg, "FLOAT");
		} else if (typeid(*arg) == typeid(OptionsArgument)) {
			add_to_synopsis(out, arg, "VALUE");
			out << "\n\t  where value=" << dynamic_cast<OptionsArgument *>(arg)->get_allowed_options();
		}
		out << endl;
		print_synopsis_string(out, 2, arg->description, 60);
		out << endl;
	}
}


