<?php
error_reporting(E_ALL);

require_once('crud_interface.php');
require_once('relational_interface.php');
require_once('orm.php');
require_once('database_connection.php');


/**
 * 
 * @author Jean-Michel Richer
 *
 */
class DAO implements CRUDInterface, RelationalInterface {
	protected $className;
	protected $tableName;
	protected $mapping;
	protected $connection;
	protected $identifier;
	
	protected $debug;
	
	/**
	 * constructor
	 * @param unknown_type $className
	 * @param unknown_type $tableName
	 * @return unknown_type
	 */
	public function __construct($className,$tableName) {
		$this->debug=true;
		
		$this->className=$className;
		$this->tableName=$tableName;
		$this->mapping=ORM::getMapping($className);
		
		$iterator=$this->mapping->values();
		while ($iterator->valid()) {
			$assoc=$iterator->current();
			if ($assoc->isIdentifier()) {
				$this->identifier=$assoc;
				break;
			}
			$iterator->next();
		}
		
		$this->connection=DatabaseConnection::getInstance()->getConnection();
		
		$this->mapping=ORM::getMapping($className);
	}
	
	public function createObject() {
		return new $this->className();
	}
	
	/**
	 * set attributes
	 * @param unknown_type $object
	 * @param unknown_type $record
	 * @return unknown_type
	 */
	private function setAttributes($object,$record) {
		$class=new ReflectionClass($this->className);
		$iterator=$this->mapping->values();
		while ($iterator->valid()) {
			$assoc=$iterator->current();
			
			$attributeName=$assoc->getAttributeName();
			$methodName="set".ucfirst($attributeName);
			if (!$class->hasMethod($methodName)) {
				throw new exception("undefined method $methodName for ".$this->className);
			}
			$method = new ReflectionMethod($this->className, $methodName);
			
   			$method->invoke($object, $record[$assoc->getColumnName()]);
			
			$iterator->next();
		}
	}
	
	private function getAttributeValue($object,$attributeName) {
		$class=new ReflectionClass($this->className);
		$methodName="get".ucfirst($attributeName);
		if ($class->hasMethod($methodName)) {
			// nothing to do
		} else {
			throw new exception("undefined method $methodName for ".$this->className);
		}
		$method = new ReflectionMethod($this->className, $methodName);
		$value=$method->invoke($object,array());
		//echo "VALUE FOR $attributeName = $value\n";
		return $value;
	}
	
	/**
	 * insert new record into database (non-PHPdoc)
	 * @see persistence/CRUDInterface#create($object)
	 */
	public function create($object) {
		$query="INSERT INTO ".$this->tableName." (";
		$columns="";
		$values="";
		
		$iterator=$this->mapping->values();
		while ($iterator->valid()) {
			$assoc=$iterator->current();
			if (!$assoc->isIdentifier()) {
				if (strlen($columns)>0) $columns.=",";
				$columns.=$assoc->getColumnName();
				if (strlen($values)>0) $values.=",";
				$value=$this->getAttributeValue($object,$assoc->getAttributeName());
				$type=$assoc->getType();
				
				if (($type=="string") || ($type=="varchar") || ($type=="text")) {
					$value="'".addslashes($value)."'";
				} else if ($type=="date") {
					$value="'".$value."'";
				} else if ($type=="timestamp") {
					$value="'".$value."'";
				} else if ($type=="float") {
					$value=sprintf("%F",$value);
				} else if (($type=="tinyint") || ($type=="int")) {
					$value=sprintf("%d",$value);
				}
				$values.=$value;
			
			}
			$iterator->next();
		}
		$query.="$columns) VALUES ($values);";
		
		if ($this->debug) {
			echo "[DAO] ".$query."\n";
		}
		$result=$this->connection->exec($query);
		if ($result===false) {
			echo __METHOD__."\n";
			echo "error: ";
			print_r($this->connection->errorInfo());
			echo "\nquery = $query";
		} else {
			$id=$this->connection->lastInsertId();
			$object->setId($id);
			$this->createRelations($object);
		}
	}
	
	/**
	 * retrieve object from its identifier and fill instance(non-PHPdoc)
	 * @param $object object to fill with data from database
	 * @see persistence/CRUDInterface#retrieve($object)
	 */
	public function retrieveByObjectId($object) {
		assert(is_object($object));
		assert(get_class($object)==$this->className);
		
		$query="SELECT * FROM ".$this->tableName." WHERE ".$this->identifier->getColumnName()."=".
			$object->getId().";";
			
		if ($this->debug) {
			echo "[DAO] ".$query."\n";
		}
			
		$result=$this->connection->query($query);
		if ($result===false) return false;
		$result->setFetchMode(PDO::FETCH_ASSOC);
		$records=$result->fetchAll();
		if (count($records)==0) return ;
		$this->setAttributes($object,$records[0]);
		$this->retrieveRelations($object);
		return $object;
	}
	
	/**
	 * retrieve object
	 * @see persistence/CRUDInterface#retrieveById($id)
	 */
	public function retrieveById($id) {
		//assert(is_integer($id));
		
		$query="SELECT * FROM ".$this->tableName." WHERE ".$this->identifier->getColumnName()."=".
			$id.";";	

		if ($this->debug) {
			echo "[DAO] ".$query."\n";
		}
			
		$result=$this->connection->query($query);
		if ($result===false) return false;
		$result->setFetchMode(PDO::FETCH_ASSOC);
		$records=$result->fetchAll();
		if (count($records)==0) return null;
		$object=$this->createObject();
		$this->setAttributes($object,$records[0]);
		$this->retrieveRelations($object);
		return $object;
	}
	
	/**
	 * retrieve all objects that correspond to given criterion (non-PHPdoc)
	 * @param $sqlWhere criterion
	 * @see persistence/CRUDInterface#retrieveAll($sqlWhere)
	 */
	public function retrieveAll($sqlWhere) {
		$query="SELECT * FROM ".$this->tableName." WHERE ".$sqlWhere.";";
		
		if ($this->debug) {
			echo "[DAO] ".$query."\n";
		}
		
		$result=$this->connection->query($query);
		
		$all=array();
		if ($result===false) return $all;
		
		$result->setFetchMode(PDO::FETCH_ASSOC);
		$records=$result->fetchAll();
		
		foreach($records as $record) {
			$object=new $this->className();
			$this->setAttributes($object,$record);
			$this->retrieveRelations($object);
			$all[]=$object;
		}
		
		return $all;
	}
	
	public function count($sqlWhere) {
		$query="SELECT COUNT(".$this->identifier->getColumnName().") ".
			"FROM ".$this->tableName." WHERE ".$sqlWhere;
		$result=$this->connection->query($query);
		$records=$result->fetchAll();
		if (!isset($records[0][0])) return 0;
		return intval($records[0][0]);
		
	}
	
	/**
	 * update given record (non-PHPdoc)
	 * @see persistence/CRUDInterface#update($object)
	 */
	public function update($object) {
		$query="UPDATE ".$this->tableName." SET ";
		$values="";
		
		$iterator=$this->mapping->values();
		while ($iterator->valid()) {
			$assoc=$iterator->current();
			if (!$assoc->isIdentifier()) {
				if (strlen($values)>0) $values.=",";
				$value=$this->getAttributeValue($object,$assoc->getAttributeName());
				$type=$assoc->getType();
				//echo "TYPE = $type\n";
				if (($type=="string") || ($type=="varchar") || ($type=="text")) {
					$value="'".addslashes($value)."'";
				} else if ($type=="date") {
					$value="'".$value."'";
				} else if ($type=="timestamp") {
					$value="'".$value."'";
				} else if ($type=="float") {
					$value=sprintf("%F",$value);
				} else if (($type=="tinyint") || ($type=="int")) {
					$value=sprintf("%d",$value);
				}
				$values.=$assoc->getColumnName()."=$value";
			}
			$iterator->next();
		}
		$query.=$values.";";
		
		if ($this->debug) {
			echo "[DAO] ".$query."\n";
		}
		
		if ($this->connection->exec($query)===false) {
			throw new exception("error: ".implode($this->connection->errorInfo()," ").$query);
			return false;
		} 
		$this->updateRelations($object);
	}
	
	/**
	 * delete record using its identifier(non-PHPdoc)
	 * @see persistence/CRUDInterface#delete($object)
	 */
	public function delete($object) {
		if (!$this->canBeDeleted($object)) return false;
		$query="DELETE FROM ".$this->tableName." WHERE ".$this->identifier->getColumnName().
			"=".$object->getid();
		
		if ($this->debug) {
			echo "[DAO] ".$query."\n";
		}
		
		$result=$this->connection->exec($query);
		$this->deleteRelations($object);
		return true;
	}
	
	public function deleteId($id) {
		$object=new $this->className();
		$object->setId($id);
		$this->retrieve($object);
		if (!$this->canBeDeleted($object)) return false;
		$query="DELETE FROM ".$this->tableName." WHERE ".$this->identifier->getColumnName().
			"=".$id;
		
		if ($this->debug) {
			echo "[DAO] ".$query."\n";
		}
		
		$result=$this->connection->exec($query);
		$this->deleteRelations($object);
	}
	
	public function deleteAll($sqlWhere) {
		$query="SELECT * FROM ".$this->tableName." WHERE ".$sqlWhere.";";
		
		if ($this->debug) {
			echo "[DAO] ".$query."\n";
		}
		
		$result=$this->connection->query($query);
		var_dump($result);
		if ($result===false) return ;
		
		$result->setFetchMode(PDO::FETCH_ASSOC);
		$records=$result->fetchAll();
		echo "[DAO] found ".count($records)." records\n";
		
		$object=new $this->className();
		foreach($records as $record) {
			$this->setAttributes($object,$record);
			$this->delete($object);	
		}
	}
	
	/**
	 * truncate table
	 * @return void
	 */
	public function truncate() {
		$query="TRUNCATE ".$this->tableName.";";
		try {
			if ($this->connection->exec($query)===false) {
				throw new exception("error: ".implode($this->connection->errorInfo()," "));
			}
		} catch(exception $exc) {
			echo __METHOD__." ".$exc->getMessage();
			echo "error: ".implode($this->connection->errorInfo()," ");
			echo "\nquery = $query";
		}
	}
	
	/**
	 * method that must be redefined if class is extended
	 * @param unknown_type $object
	 * @return unknown_type
	 */
	public function createRelations($object) {
		
	}
	
	public function updateRelations($object) {
		
	}
	
	public function retrieveRelations($object) {
		
	}
	
	public function deleteRelations($object) {
		
	}
	
	public function canBeDeleted($id) {
		return true;
	}
} 
?>